﻿:Namespace Anova
(⎕IO ⎕ML ⎕WX)←1 1 3

 alias←{
     ⍝∇ Generale aliases for fractional factorial designs
     ⍝⍺ List of aliases
     ⍝⍵ Standard Order
     ⍝← Matrix of aliases
     ⎕IO←0 ⋄ ⎕ML←1
     ⍺≡'I':⍵
     SO←{2=≡⍵:⍵~'I'(,'I')         ⍝ d.g. 'I' 'A' 'B' 'AB' ...
         0=⊃0⍴⍵:1↓standardOrder ⍵↑⎕A  ⍝ e.g. 3
         1↓standardOrder ⍵}⍵           ⍝ 'ABCD'
         ⍝A←⊆⍺
     A←'='toNestedVector ⍺
     f←∪~∩                             ⍝ Multiplication of Effects
     I←⊂''                             ⍝ Plus
     G←∪I,f pairwise A,⍨I              ⍝ Generators
     M←{⍵[⍋⍵]}¨SO∘.f G                 ⍝ Matrix of aliases
     Z←{⍵[⍋≢¨⍵]}across M               ⍝ Sort by # of interactions
     Z
         ⍝ Z[⍋(≢,⊢)¨Z[;0];]                   ⍝ Sort by main effects, then alphabetically
 }

 anovaDF←{
     ⍝3-way Anova
     ⍝⍵:  n-dimensional array
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ N←⍴⍵
     DF←¯1+¯1↓N                ⍝ Main Effects
     DF,←×pairwise DF          ⍝ Two-way Interactions
     DFT←¯1+×/N                ⍝ Total degrees of freedom
     DFE←×/N-(-≢N)↑1           ⍝ Error    "     "    "
     DF,0~⍨(DFT-+/DF,DFE),DFE  ⍝
     
 }

∇ AZ←NL anovaR D;A;B
      ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
      ⍝∇ Perform ANalysis Of VAriance (ANOVA) using R
      ⍝∊ anova
      ⍝⍺ NL: Name List
      ⍝⍵ NS: Database
      ⍝⍵ [A1: Factor 1]
      ⍝⍵ [A2: Factor 2]
      ⍝← NS:   Namespace containing the following variables
      ⍝        AnovaTable  RSq  RSqAdj  S  Type
      ⍝⍕ NS ← anova X [Av1 [Av2]]
      ⍝⍎ report anova D.Environment D.Party     ⍝ One-way ANOVA
      ⍝⍎ report anova D 'Environment Party'     ⍝ One-way ANOVA using factor name
      ⍝⍎ anova MPG AUTO GAS                     ⍝ Two-way ANOVA
 D←{U←⊂(0≢↑0⍴⍵)/'"' ⋄ 'c(',(1↓∊U{','⍺,(⍕⍵),⍺}¨⍵),')'}¨D
 _←##.∆r.x¨NL{⍺,'<-',⍵}¨D
      ⍝NL ##.∆r.p¨D
 A B←NL
 _←##.∆r.x∊'XX <- data.frame('A','B')'
 Z←##.∆r.f∊'summary(aov('A'~'B',data=XX)'
∇

 anovaResiduals←{
     ⍝ ⍵ n-dimensionalmatrix last dim residuals
     ⍝ ← Vector of residuals
     ⎕ML←1 ⋄ ⎕IO←0
     M←mean,⍵
     EFF←M-⍨(mean across{⍺ ⍵}mean down)mean across ⍵
     
     
     YHAT←M+⊃∘.+/EFF
     ⍵-⍤1 0⊢YHAT
          ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     Y←(sum across ⍵)÷⊃⌽⍴⍵
     EFF←Y-M
     ⍵-⍤1 0⊢EFF
     
     
 }

 anovaSS←{
     ⍝∇ N-way Anova SS
     ⍝∊ (c) 2022, Stephen Mansour
     ⍝⍺  Optional:  Levels Deep
     ⍝⍵  n-dimensional array
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ N←⍴⍵ ⋄ TOL←1E¯10
     YN←(2*⍨+/,⍵)÷×/,N                        ⍝ Correction Factor
     SST←(+/,⍵*2)-YN                          ⍝ Total sum of squares
     FA←¯1+≢N                                 ⍝ Factors
     Y←+/⍣FA¨({(-⍵)⌽¨⊂⍵}⍳FA+1)⍉¨⊂⍵            ⍝ Main-Effect Sums
     SSA←¯1↓YN-⍨(Y+.*¨2)÷(×/÷⊢)N              ⍝ Main-Effect SS
     I←⍋∘⍒¨(⍳FA+1)∘∊¨,pairwise⍳FA             ⍝ Transpose vectors
     Y←,¨+/⍣(FA-1)¨I⍉¨⊂⍵                      ⍝ 2-way Interaction Sums
     SSAB←(Y+.*¨2)÷{(×/⍵)÷×pairwise ¯1↓⍵}N    ⍝
     SSAB-←YN++pairwise SSA                   ⍝ 2-way Sum of Squares
     SSABC←SST-+/SSA,SSAB                     ⍝ 3+way Sum of Squares
     1=R←⊃⌽N:SSA,SSAB,(TOL<SSABC)/SSABC       ⍝ Replicates?
     SSE←(+/,var across ⍵)×R-1                ⍝ Separate Error from 3+way
     SSA,SSAB,(TOL>|SSE-SSABC)↓(SSABC-SSE),SSE ⍝ List them
 }

 anovaSource←{
     ⍝3-way Anova
     ⍝⍵:  n-dimensional array
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ N←⍴⍵
     Eff←,¨(¯1+≢N)↑⎕A        ⍝ Main Effects
     Eff,←,pairwise Eff      ⍝ Two-way Interactions
     Eff,←(4>≢N)↓⊂'ABC'       ⍝ Include 3-way, error
     1=⊃⌽N:(¯1↓Eff),⊂'Err'   ⍝ If 1 replicate, Int=Error
     Eff,⊂'Err'              ⍝ Else append Error
 }

 anovaSummary←{
         ⍝∊ ANOVA
     ⎕IO←0 ⋄ ⎕ML←3
     n←⍵.count
     x←⍵.mean
     s←⍵.sdev
     NMS←⍵{0::⍵ ⋄ ⍺.name}¨(≢⍵)↑⎕A
     X←(n+.×x)÷+/n
     SST←n+.×(x-X)*2
     SSE←(n-1)+.×s*2
     DFT←¯1+≢x
     DFE←(+/n)-DFT+1
        ⍝  'A' 'Error',2 2⍴SST DFT SSE DFE
     'A' 'Error',2 3⍴SST DFT 2 SSE DFE 0
     ns←'Treatments'buildAnovaTable 2 2⍴SST DFT SSE DFE
     ns.Type←'ANOVA'
     ns.Means←'Treatment' 'N' 'Mean' 'Sdev'⍪NMS,n,x,[0.5]s
     ns}

 anova_report←{0::⍵._Report ⋄
     ⎕IO←0 ⋄ ⎕ML←3
         ⍝ DF←(¯1∘↓,¨↑∘⌽)¯1↓1↓⍵.AnovaTable[;2]
     DF←⍵.DF2,⍨¨¯2↓1↓⍵.AnovaTable[;2]
     DF←(~¯2↓1↓⍵.AnovaTable[;0]∊'Blocks' 'Rows' 'Columns')/DF ⍝ Don't calc F for blocks
     H←⊂¯9↑' '~⍨,'K3G<F(⍺=.99Z)>'⎕FMT ⍺   ⍝ Header for critical Value
     FC←(∪DF)fDist criticalValue<¨⍺    ⍝ Append F(⍺) to table
     TBL←⍵.AnovaTable,H,(¯1+≢⍵.AnovaTable)↑FC[DF⍳⍨∪DF]
     RPT←⍵._Report{⍺,(≢⍺)↑' '⍪¯1⊖(↑⍵)⍪'-'⍪'BF9.2'⎕FMT 1↓⍵}TBL[;6]
          ⍝4<≢⍵.AnovaTable:⍵._Report                   ⍝ Append F(⍺) to Report
     4<≢TBL:RPT                              ⍝ Append F(⍺) to Report
     U←stats¨↓{n m s←↓⍉⍵ ⋄ N←1+n+.-1         ⍝ Adjust "sample size" and
         vp←((s*2)+.×n-1)÷N-1                ⍝ Pooled var to tie confints
         N,m,[0.5]sqrt vp×N÷n}1 1↓⍵.Means    ⍝ to Minitab for anova
         ⍝D←(1-⍺)1(1↓⍵.Means[;0])mean formatConfInt U
     D←(1-⍺)'U'(1↓⍵.Means[;0])mean formatConfInt U
                ⍝  LSD←'Treatment' 'Mean Diff' 'DF' 'SE' 'T' 'P' 'LSD'⍪lsd ⍵
     LSD←⍺ lsd ⍵
     TXT←⍕,[⍳0]RPT' '⍵.Means' ' 'Confidence Interval' ' 'D.Output' 'LSD
     120↑[1]TXT}

 anovablock←{
      ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
      ⍝∇ Perform Blocked ANalysis Of VAriance (ANOVA)
      ⍝∊ ANOVA
      ⍝⍵ X:  Response (simple or partitioned array
      ⍝⍵ NS: Database
      ⍝⍵ [A1: Factor 1]
      ⍝⍵ [A2: Factor 2]
      ⍝← NS:   Namespace containing the following variables
      ⍝        AnovaTable  RSq  RSqAdj  S  Type
      ⍝⍕ NS ← anovablock X [Av1 [Av2]]
      ⍝⍎ report anovanlock D.Environment D.Party  ⍝ One-way ANOVA
      ⍝⍎ report anova D 'Environment Party'       ⍝ One-way ANOVA using factor name
     ⎕IO←0 ⋄ ⎕ML←3                           ⍝ Set system variables
     d←1 1↓⍵                                 ⍝ Remove row names and columns
     NL←1↓¨(⍵[;0])(⍵[0;])                    ⍝ Get name lists
     y n←(sum,count),d                       ⍝ Get sum and count
     a b←⍴d                                  ⍝ Number of levels for each factor
     yi yj←(+⌿{⍺ ⍵}+/)d                      ⍝ Sum columns and rows
         ⍝yn←(y*2)÷n                              ⍝
     ss←+/,d*2                               ⍝ Total sum of squares
     ss,←+/((+/d)*2)÷b                       ⍝ Row sum of squares
     ss,←+/((+⌿d)*2)÷a                       ⍝ Column sum of squares
     SS←ss-(y*2)÷n                           ⍝ Adjusted sum of squares
     SS←1↓SS,(↑SS)-+/1↓SS                    ⍝       "   "   "   "
     DF←{⍵,×/⍵}a b-1                         ⍝ Degrees of freedom
     ns←'Treatments,Blocks'buildAnovaTable SS,[0.5]DF ⍝
     ns.Type←'ANOVA'                         ⍝ Result is ANOVA object
     ns}

 aov←{
      ⍝⍟ Copyright (C)  2022 by Stephen M. Mansour
      ⍝∇ Generate factorial design
      ⍝∊ ANOVA
      ⍝⍵ X:    Response - n-dimensional numeric vector; trailing dimension is replicates
      ⍝← Z:   Table containing the following columns [Source|SS|DF]
      ⍝        AnovaTable  RSq  RSqAdj  S  Type
      ⍝⍕  ←  factorialDesign X
      ⍝⍎ 1 factor 4 4⍴98 97 99 96 91 90 93 92 96 95 97 95 95 96 99 98
      ⍝⍎ 2 factor 3 3 4⍴130 155 74 180 34 40 80 75 20 70 82 58 150 188 159 126 136 122 106 115 25 70 58 45 138 110 168 160 174 120 150 139 96 104  82 60
      ⍝⍎ 3 factor 3 2 2 2⍴ ¯3 ¯1 ¯1 0 ¯1 0 1 1 0 1 2 1 2 3 6 5 5 4 7 6 7 9 10 11
      ⍝⍎ factor Data Factor1 ... FactorN
      ⍝⍎ factor NS  Data 'Factor1 ... FactorN'
     ⎕IO←0 ⋄ ⎕ML←1 ⋄
     W NMS←{isNamespace⊃⍵:(toArray⊃⍎/⍵)(1↓' 'toNestedVector⊃⌽⍵)
         NMS←(¯1+≢⍵)↑⎕A  ⍝ Factor Names
         (1=⍴⍴⍵)∧2≤|≡⍵:(toArray ⍵)NMS ⋄ ⍵ 0J1}⍵
     
     R←⊃⍴⍴W ⋄ ⍺←R-1 ⋄ NL←~1 2 3 4∊⍨⊃⊃⍺ ⋄
     A←≢⍣NL⊢⍺ ⋄ NMS←NL⊃NMS ⍺
     NMS←(NMS≡0J1)⊃NMS(A↑⎕A)
     A=R+1:latinSquare W
     A=R+2:graecoLatinSquare W
     A=1:oneWay W
     W←A{R←⊃⍴⍴⍵ ⋄ ⍺=R-1:⍵ ⋄ ⍺=R:(1⌽1,⍴⍵)⍴⍵ ⋄ ⍺<R-1:,[⍺↓⍳R]⍵
         ⎕SIGNAL'Nonce Error'}W ⍝ Reshape right arg
     X←+/W
     L R←¯1(↓{⍺ ⍵}↑)⍴W ⋄ K←≢L ⋄ N←×/L,R
     Y2N←N÷⍨(+/,W)*2
     SST←(sumSquares,W)-Y2N
          ⍝⍝⍝⍝⍝ Main Effects ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     XA←K{⍺=2:+/¨⍵(⍉⍵)
         ⍺=3:+/⍣2¨(0 1 2)(2 0 1)(1 2 0)⍉¨⊂⍵
         ⍺=4:+/⍣3¨(0 1 2 3)(3 0 1 2)(2 3 0 1)(1 2 3 0)⍉¨⊂⍵
         'Nonce Error'}X
     SSA←Y2N-⍨N÷⍨L×sumSquares¨XA
          ⍝⍝⍝⍝⍝ 2-way Interactions ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     XAB←K{⍺=2:,¨⍵
         ⍺=3:,¨+/¨(0 1 2)(0 2 1)(2 0 1)⍉¨⍵
         ⍺=4:+/⍣2¨(0 1 2 3)(0 2 1 3)(2 0 1 3)(3 0 1 2)(3 0 2 1)(3 2 0 1)⍉¨⍵}⊂X
     D←(N÷⍨×pairwise L)×sumSquares¨XAB
     SSAB←D-Y2N++pairwise SSA
          ⍝⍝⍝⍝⍝⍝ Error Sum of Squares ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     SUBT←Y2N-⍨R÷⍨sumSquares,X
     SSE←SST-SUBT
     SSABC←SST-+/SSA,SSAB,SSE   ⍝ 3+ interactions
          ⍝⍝⍝⍝⍝ Assemble Data ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     SS←SSA,SSAB,SSABC,SSE,SST
     Source←2⌽'Error' 'Total',{⍵,(,pairwise ⍵),⊂⍵}NMS
         ⍝Source←2⌽'Error' 'Total',{⍵,({⍺ ⍵}pairwise ⍵),⊂⍵}NMS
     DFE←×/L,R-1
     MSE←SSE÷DFE
     DF←(L-1),×pairwise L-1
     DF,←N-1++/DF,DFE      ⍝ Include ABC
     DF,←DFE,N-1
     ¯1↓(DF≠0)⌿Source,SS,⍪DF
 }

 blocked←{
      ⍝⍟ Copyright (C) 2022 by Stephen M. Mansour
      ⍝∇ Generate blocked design
      ⍝∊ ANOVA
      ⍝⍵ DataMatrix       X  Response - n-dimensional numeric vector; trailing dimension is replicates
      ⍝← Source           C  Description
      ⍝← SumOfSquares     Xv
      ⍝← DegreesOfFreedom Nv
      ⍝← DenominatorIndex Nv Row position of denomator mean square
      ⍝⍕ NS ←  blocked X
      ⍝⍎ blocked 4 3⍴98 97 99 96 91 90 93 92 96 95 97 95 95 96 99 98     ⍝ Balanced Design
      ⍝⍎ blocked 4 3⍴98 97 99 96 91 90 93 92 96 ⎕null 97 95 95 96 99 98  ⍝ Missing Data
      ⍝⍎ blocked   4 4⍴.1×93 94 96 100 94 93 98 99 92 94 95 97 97 96 100 102
      ⍝⍎ factor Data Factor1 ... FactorN
      ⍝⍎ factor NS  Data 'Factor1 ... FactorN'
     ⎕IO←0 ⋄ ⎕ML←1 ⋄
     W NMS←{isNamespace⊃⍵:(toArray⊃⍎/⍵)(1↓' 'toNestedVector⊃⌽⍵)
         NMS←(¯1+≢⍵)↑⎕A  ⍝ Factor Names
         (1=⍴⍴⍵)∧2≤|≡⍵:(toArray ⍵)NMS ⋄ ⍵ 0J1}⍵
     W←{~⎕NULL∊⍵:⍵                 ⍝ If missing values
         I←⍸⍵=⎕NULL                ⍝ Find nulls
         YIJ←(+/,+⌿)(0@I)⍵         ⍝ Row, Column Sums
         YT←+/(≢⍵)↑YIJ             ⍝ Total
         X←YIJ[↑I+⊂0,≢⍵]+.×AB←⍴⍵   ⍝
         X←(X-YT)÷×/AB-1           ⍝ Denominator
         (X@I)⍵}W
     EFF SS DF←↓⍉(≢⍴W)aov W        ⍝ Calculate factorial
     (¯1↑DF)-←+/,⍵∊⎕NULL           ⍝ Remove
     Source←'Treatments' 'Blocks' 'Error'    ⍝ One-Factor blocked
     2=⍴⍴W:⍉↑Source SS DF(≢SS)
     Source←'Rows' 'Columns' 'Interaction' 'Blocks' 'Error'  ⍝ 2 factor blocked
     D←'RCBIEEE'{⍺,+⌿⍵}⌸SS,⍪DF
     Source,0 1↓D[(⊃¨Source)⍳D[;0];],5↑3/≢D
     
     
     
 }

 buildAnovaConfInt←{
      ⍝∇ Calculate Confidence Interval for Anova
     ⎕ML←3 ⋄ ⎕IO←0
     l n m s←↓⍉⍵.Table ⋄
     N←1+n+.-1    ⍝ Adjust "sample size" and
     vp←((s*2)+.×n-1)÷N-1     ⍝ Pooled var to tie confints
     A←N,m,[0.5]sqrt vp×N÷n   ⍝ Stats
     U←stats¨↓A               ⍝ Create Stats objects
     ⍵.Results←⊃mean confInt¨U   ⍝
     ⍵.PointEstimate←m
     ⍵.ResultsLabels←l
     ⍵
 }

 buildAnovaTable←{
         ⍝⍟ Copyright (C) 2018, 2022 by Stephen M. Mansour  - #.TamStat.Anova
         ⍝∇ Create anova table from SS and DF
         ⍝∊ Util
         ⍝⍺ FactorNames   'A,B'
         ⍝⍵ 3 or 5 col Matrix [SS|DF|[J|[C|EFF]] (No totals)
         ⍝← Formatted Anova Table
          ⍝⍝⍝⍝⍝⍝⍝⍝⍝ Calculate the table ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ ⎕DIV←1 ⋄ ⍺←⍬          ⍝ Set system variables
     80 160 320∊⍨⎕DR↑⍵:⍵[;0]∇ ⍵[;1 2]      ⍝ If 1st item character
         ⍝ formatAnova←{T←⍵                      ⍝ Format anova table
     ⍝         T[;4]←{⍵⌈0.000009××⍵}T[;4]        ⍝
     ⍝         D←0⌈5-⌈10⍟⌈⌿T[;0],0.000001        ⍝ Decimal Places for SS
     ⍝         FW←16 6 16 10 10                  ⍝ Field Widths
     ⍝         SSF←'C',(×D)⊃'I16'('F16.',⍕D)     ⍝ SS format
     ⍝         H←∊(-FW)↑¨⍺                       ⍝ Headers
     ⍝         U←(,1,[0.5]FW-1)/10⍴' -'          ⍝ Underlines
     ⍝         FS←SSF,',I6,B',SSF,',BF10.2,'     ⍝ Format String
     ⍝         FS,←'BO0.000009⍞<0.00001⍞F10.5'   ⍝ P-Value format
     ⍝         T←H⍪U⍪FS ⎕FMT T                   ⍝ Format it
     ⍝         ¯1(↓{⍺⍪U⍪⍵}↑)T}                   ⍝ Include total line
          ⍝⍝⍝⍝⍝⍝ Format Row Heading ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     formatRowHeader←{
         NMS←,¨'A',(2<≢⍵)/'B' 'AB'         ⍝ Default Names
         FA←(0=≢⍺)⊃(toNestedVector ⍺)NMS   ⍝
         FA←(~∨\¨1 0 0 0∘⍷¨FA='+')/¨FA     ⍝ Remove 3+ ways from aliases
         RH←(-1+≢⍵)↑'Interaction' 'Error' 'Total'
         RH[⍳≢FA]←FA                       ⍝ Insert Factor Names
         RH}
          ⍝⍝⍝⍝⍝⍝ Format Other Data ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     formatData←{
         FS←'<S = >,F10.5,<  R-Sq = >,K2Q<%>F7.2,<  R-Sq(adj) = > ,K2F7.2,<%>'
         FS ⎕FMT 1 3⍴⍵}
          ⍝⍝⍝⍝⍝⍝ Put it all together ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     NS←⎕NS''                                ⍝ Create namespace
     4=↑⌽⍴⍵:⍺ fractionalFactorial ⍵          ⍝ Fractional Factorial
     A W←⍺{
         (0=≢⍺)∨×⍵[¯1+≢⍵;1]:⍺ ⍵            ⍝ If DFE=0,
         B←3>≢¨⍺ ⋄ A←(B/⍺),⊂'Error'          ⍝ use 3+ way interactions and
         W←B{(⍺⌿⍵,1++/⍺)⍪1⌽0,⍵+.×⍨~⍺}⍵[;0 1] ⍝ Sum SS and DF for error
         A W}⍵                               ⍝
     TBL R←A calcAnova W                     ⍝ Calculate the table
     0=≢A:TBL                                ⍝
     H←toNestedVector'SOURCE,SS,DF,MS,F,P'   ⍝ Header names
     FT←(1↓H)formatAnova TBL                 ⍝ Format ANOVA table
     RH←A formatRowHeader W                  ⍝ Format row headers
     FD←formatData R                         ⍝ Format other data
     RP←¯1(↓{⍺⍪'-'⍪⍵}↑)⊃(↑H)'------',RH      ⍝ Include total line
     NS.AnovaTable←H⍪RH,TBL
     NS.(S RSq RSqAdj)←R×1 100 100
     NS._Report←⍕,[⍳0]'ANOVA Table' ' '(RP,FT)' 'FD  ⍝ _Report property
     NS                                      ⍝ Result is namespace
 }

 calcAnova←{
          ⍝∇ Calculate anova table and r-Squared values
          ⍝⍵ 3-column matrix [SS|DF|J]   (Denominator index)
     ⎕IO←0 ⋄ ⎕ML←1                  ⍝ Set system variables
     SS DF J←↓⍉⍵                    ⍝ unpack data
     MS←SS÷DF                       ⍝ Mean Squares
     I←¯1+J~0                       ⍝ Index of denomator
     B←J≠0                          ⍝ Test items
     F←(B/MS)÷MS[I]                 ⍝ Calculate F statistics
     P←((B/DF),¨DF[I])fDist prob>¨F ⍝ Get P values
     F P←B∘\¨F P                    ⍝ Expand to fit
     T←+⌿SS,⍪DF                     ⍝ Totals
     Z←(⍉↑SS DF MS F P)⍪5↑T         ⍝ Build Anova Table
     E←,¯1 2↑⍵                      ⍝ Error values
     R2←1-E÷⍥⊃T                     ⍝ r-Squared
     R2A←1-÷/(2↑E)÷T                ⍝ r-Squared Adjusted
     S←(⊃⌽MS)*÷2                    ⍝ std error
     Z(S R2 R2A)                    ⍝ Result
 }

 calcNested←{
     ⍝⍺ Source
     ⍝⍵ SS, DF
     ⍝← Anova Table
     ⎕IO←0 ⋄ ⎕ML←1
     MS←÷/⍵                            ⍝ Mean Square
     S←(⊃⌽MS)*÷2                       ⍝ Error
            ⍝  F←¯1↓MS÷↑⌽MS                      ⍝ Test Statistic (F)
     F←⍺{3=≢⍵:2÷/⍵
         WI←'('∊¨⍺
         ¯1↓⍵÷⍵[4 2 5 4 5 5]
     }MS
     pF←fDist probability>             ⍝ Upper-tail Probability
     P←⎕CT⌈({¯1↓⍵,¨⊃⌽⍵}⍵[;1])pF¨F      ⍝ P-Value
     T←⍵⍪+⌿⍵                           ⍝ Include totals
     R2←1-÷/¯2↑T[;0]                   ⍝ r-Squared
     R2A←1-÷/÷/¯2 2↑T                  ⍝ r-Squared Adj
     T←T,0⍪⍨⍉↑MS F P                   ⍝ Build ANOVA Table
     T(S R2 R2A)}

 calcw2←{
     ⍝∇ Calculate Variances/⍵^2 given Anova Table
     ⍝⍵ Anova Table
     ⍝← Vector of Variances
     ⍝← Vector of Omega2
     ⎕IO←0 ⋄ ⎕ML←1
     E T←↓¯2↑⍵        ⍝ Error, Total Rows
     DF←1↓⍵[;2]       ⍝ Degrees of Freedom
     MS←¯2↓1↓⍵[;3]    ⍝ Mean Square Between
     MSW←3⊃E          ⍝ Mean Square Error
     SST←1⊃T
     N←1+⊃⌽DF         ⍝ Sample Size
     S2←(¯2↓DF)÷N÷MS-MSW
     S2T←+/S2,MSW
     S2÷S2T
     ⍝         NS.Omega2←(⍵[1;1]-⍵[1;2]×⍵[2;3])÷⍵[3;1]+⍵[2;3]
     
     
     
     
 }

 factor←{
      ⍝⍟ Copyright (C) 2022 by Stephen M. Mansour
      ⍝∇ Generate 1, 2 or 3-factor model
      ⍝∊ ANOVA
      ⍝⍺ Number of Fixed Factors
      ⍝⍺ Number of Random Factors
      ⍝⍵ X:    Response - n-dimensional numeric vector; trailing dimension is replicates
      ⍝← TBL:  [Source|SS|DF|DF2I]
      ⍝⍕ NS ←  randomFactor X
      ⍝⍎ X←130 155 74 180,34 40 80 75, 20  70 82 58
      ⍝⍎ X,←150 188 159 126, 136 122 106 115, 25 70 58 45
      ⍝⍎ X,←138 110 168 160, 174 120 150 139, 96 104 82 60
      ⍝⍎ 2 factor 3 3 4⍴X
      ⍝⍎ 2 factor 2 2⍴↓X
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ A←¯1+⊃⍴⍴⍵ ⋄ ⍺←A
     A←(0=≢⍺)⊃⍺ A             ⍝ If not specified, use rank-1
     B←{1<≡⍵:⍵ ⋄ +/⍵}A
     EFF SS DF←3↑↓⍉B aov ⍵
     I←(0 2)(1 1)⍳⊂A          ⍝ 0=Random, 1=Mixed, 2=Fixed
     J←I⊃(3 3 4)(3 4 4),⊂{⍵⍴⍨¯1+⍵}≢DF
     ⍉↑EFF SS DF J
     
     
 }

 factorial2k←{
     ⍝∇  Generate a 2K factorial design
     ⍝⍺  [Alias←'I'] Cv Effects equivalent to Identity I
     ⍝⍵  V   Xv List of observed values in standard order
     ⍝⍵  M   Xm Rows=TreatmentCombinations in standard order;  Columns=Replicates
     ⍝←  EFF Xm Rows=Effects; Columns: EffectName EffectValues DegreesOfFreedom DenominatorIndex
     ⍝⍕  EFF←[Alias] factorial2k V|M
     ⍝⍎  factorial2k 4 3⍴28 25 27 36 32 32 18 19 23 31 30 29
     ⍝⍎  factorial2k 8 2⍴¯3 ¯1 ¯1 0 ¯1 0 1 1 0 1 2 1 2 3 6 5
     ⍝⍎  'ABC' factorial2k 4 2⍴1037 1052 669 650 633 601 729 860
     ⍝⍎  'ABD=ACE' factorial2k  6 9 35 50 18 22 40 63
     ⍺←'I' ⋄ ⎕IO←0 ⋄ ⎕ML←1     ⍝ Determine
     f←{I←⍋⍵[;0]⍳¨'='          ⍝ Sort by main effects, interactions
         EFF SS DF←↓⍉⍵[I;]
         J←{¯1↓⍵/⍵}≢SS
         ⍉↑EFF SS DF J}
     K R Y←{D←⍴⍵               ⍝ Factors(K), replicates(R)
         (2⍟D[0]),D[1],⊂+/⍵}⍪⍵ ⍝ and Sums (Y).
     H←hadamard K              ⍝ Matrix of (+,-)1's
     C←Y+.×H                   ⍝ Contrasts
     SS←1↓(C*2)÷R×2*K          ⍝ Sums of Squares
     EFF←C÷R×2*K-1             ⍝ Effects
     SO←1↓standardOrder K↑⎕A   ⍝ Effect Names
     SO←{⍺,'+',⍵}/⍪⍺ alias SO
     J←⍋{(≢,⊢)⍵↑⍨⍵⍳'+'}¨SO     ⍝ Sort by main effects, then alphabetically
     N←≢,⍵                     ⍝ Sample Size
     SST←(sumSquares,⍵)-(2*⍨sum,⍵)÷N ⍝ Total sum of squares
     R>1:f(1⌽1,(SO,⍪SS)[J;])⍪'Error'(SST-+/SS)(N-1+≢SO) ⍝ If Replicates, calc SSE
     ⍺≢'I':(SO,SS,1,1↓C,⍪EFF)[J;]
     Error←(1+8<≢SO)<SO⍳¨'+'
     (Error/SO)←⊂'Error'       ⍝ Count as sampling error
     f SO{⍺,+⌿⍵}⌸⌽1,⍪SS        ⍝ Consolidate error SS and DF
 }

 formatAnova←{⎕ML←3 ⋄ ⎕IO←0 ⋄ ⎕DIV←1
     T←⍵                               ⍝ Format anova table
     T[;4]←{⍵⌈0.000009××⍵}T[;4]        ⍝
     D←0⌈5-⌈10⍟⌈⌿T[;0],0.000001        ⍝ Decimal Places for SS
     FW←16 6 16 10 10                  ⍝ Field Widths
     SSF←'C',(×D)⊃'I16'('F16.',⍕D)     ⍝ SS format
     H←∊(-FW)↑¨⍺                       ⍝ Headers
     U←(,1,[0.5]FW-1)/10⍴' -'          ⍝ Underlines
     FS←SSF,',I6,B',SSF,',BF10.2,'     ⍝ Format String
     FS,←'BO0.000009⍞<0.00001⍞F10.5'   ⍝ P-Value format
     T←H⍪U⍪FS ⎕FMT T                   ⍝ Format it
     ¯1(↓{⍺⍪U⍪⍵}↑)T}

 fractionalFactorial←{
     ⍝⍺:  Alias Structure
     ⍝⍵:  5-column matrix
     ⍝←:  Namespace
     ⎕IO←0 ⋄ ⎕ML←1
     NS←⎕NS''
     SS DF C EFF←↓⍉⍵  ⍝ Unpack
     FA←(~∨\¨1 0 0 0∘⍷¨⍺='+')/¨⍺
     V←{1⌽'][',(⍵⍳'+')↑⍵}¨⍺
       ⍝   NS.Plot←normProbPlot show EFF
     A←'*'@{2<≢¨⍵}(⍺⍳¨'+')↑¨⍺
     NS.Plot←(A normProbPlot EFF).Output
     H←'Variable' 'Alias' 'Contrast' 'Effect' 'SS' 'Pct'
     PCT←100×SS÷+/SS
     NS.Effects←H⍪V,⍺,0.001 round C,EFF,SS,⍪PCT
     NS._Report←⍕⍪'Fractional Factorial Design'⍬ NS.Effects ⍬'Normal Probability Plot'NS.Plot
     NS
 }

 hadamard←{
      ⍝∇ Generate  hadamard matrix
      ⍝∊ ANOVA
      ⍝⍺ [Symbols←1,¯1] Xv  2-item vector
      ⍝⍵ Degree         Ns  Number
      ⍝← Matrix         Xm  Degree x Degree matrix of symbols
      ⍝⍕  Matrix←[Symbols] hadamard Degree
      ⍝⍎  hadamard 3
      ⍝⍎  1 0 hadamard 5
      ⍝⍎  hadamard show 4
     ⎕IO←0 ⋄ ⍺←1 ¯1
     ~0J1∊⍺:⍺[{(⍵,~⍵)⍪⍵,⍵}⍣⍵⊢⍪0]
     Z←'+-'∇ ⍵
     TC←standardOrder ⍵↑⎕C ⎕A
     EFF←standardOrder ⍵↑⎕A
     ⍕(' ',TC),EFF⍪Z
 }

 latinSquare←{
     ⍝∇  Latin Square design
     ⍝⍟  Copyright 2023 by Stephen M. Mansour
     ⍝∊  ANOVA
     ⍝⍺  Latin Square pattern (Default - Cyclical)
     ⍝⍵  n×n Matrix
     ⍝←  Namespace
     ⍝⍎  latinSquare 5 5⍴¯1 ¯5 ¯6 ¯1 ¯1 ¯8 ¯1 5 2 11 ¯7 13 1 2 ¯4 1 6 1 ¯2 ¯3 ¯3 5 ¯5 4 6
     ⎕IO←0 ⋄ ⎕ML←1
     P←≢⍵                      ⍝ Size
     ⍺←(⍳P)⌽P P⍴P↑⎕A           ⍝ Default is cyclical
     2=≡⍵:⍺ ∇⊃¨⍵               ⍝ If nested, recursive
     YN←÷/(+/,⍵)P*2            ⍝
     LS←P{0=≢⍵:(⍳⍺)⌽⍺ ⍺⍴⍺↑⎕A   ⍝ If not specified use default
         ∧/,⍵∊⎕A:⍵             ⍝ If Alphabetical, OK
         ⎕A[⍵⍳⍨∪(,⍵)[⍋,⍵]]}⍺   ⍝ Else make Alphabetical
     D←⍵∘{+/,⍺×LS=⍵}¨P↑⎕A      ⍝ Latin Square Grouping
     SS←P÷⍨D(+/⍵)(+⌿⍵)+.*¨2    ⍝ Sums of Squares
     SS←SS,⍨+/,⍵*2             ⍝ Total sum of Squares
     SS-←÷/(+/,⍵)P*2           ⍝ Correction Factor
     SS,←(⊃SS)-+/1↓SS          ⍝ Sum of Squares Error
     DF←(+/,⊢)3 1/×\P-1 2      ⍝ Degrees of Freedom
     H←'Treatments' 'Rows' 'Columns' 'Error'
     ⍉↑H(1↓SS)(1↓DF)4          ⍝ Output
     
     
 }

 lsd←{
     ⍝∇ Least Significant Difference
     ⍝∊  ANOVA
     ⍝⍺:  Significance level
     ⍝⍵:  Namespace result of anova function
     ⎕IO←0 ⋄ ⎕ML←3
     ⍺←0.05
     G N M←↓⍉1↓⍵.Means[;0 1 2]
     DIF←-pairwise M
     DF MSE←(⊖⍵.AnovaTable)[1;2 3]
     SE←sqrt MSE×+⍥÷pairwise N
     T←DIF÷SE
     P←2×DF tDist prob>|T
     H←{⍺,' vs ',⍵}pairwise G
     LSD←SE×DF tDist critVal<⍺÷2
     Z←H,0.0001 round DIF,1,SE,T,P,⍪LSD
     'Treatment' 'Mean Difference' 'DF' 'SE' 'T' 'P' 'LSD'⍪Z
 }

 mixedFactor←{
      ⍝⍟ Copyright (C) 2022 by Stephen M. Mansour
      ⍝∇ Generate 2 factor mixed  model  A=Fixed, B=Random
      ⍝∊ ANOVA
      ⍝⍵ X:    Response - n-dimensional numeric vector; trailing dimension is replicates
      ⍝← TBL:  [Source|SS|DF|MS|F|P]
      ⍝⍕ NS ←  mixedFactor X
      ⍝⍎ X←130 155 74 180,34 40 80 75, 20  70 82 58
      ⍝⍎ X,←150 188 159 126, 136 122 106 115, 25 70 58 45
      ⍝⍎ X,←138 110 168 160, 174 120 150 139, 96 104 82 60
      ⍝⍎ nested 3 3 4⍴X
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ A←¯1+⊃⍴⍴⍵ ⋄ ⍺←A
     A←(0=≢⍺)⊃⍺ A
     EFF SS DF←↓⍉A factor ⍵
     MS←SS÷DF
     J←2 3 3
     F←MS[⍳3]÷MS[J]
     DF12←DF[⍳3],¨DF[J]
     P←DF12 fDist prob>¨F
     EFF,SS,DF,MS,(F,0),⍪P,0
 }

 nested←{
      ⍝⍟ Copyright (C) 2022 by Stephen M. Mansour
      ⍝∇ Generate nested design
      ⍝∊ ANOVA
      ⍝⍵ X:    Response - n-dimensional numeric vector; trailing dimension is replicates
      ⍝← NS:   Namespace containing the following variables
      ⍝        AnovaTable  RSq  RSqAdj  S  Type
      ⍝⍕ NS ←  nested X
      ⍝⍎ nested 3 2 3⍴20 18 14 19 20 20 14 18 14 12 12 9 13 16 13 9 4 4
      ⍝⍎ X2←1 ¯1 0 ¯2 ¯3 ¯4 ¯2 0 1 1 4 0 , 1 ¯2 ¯3 0 4 2 ¯1 0 ¯2 0 3 2,2 4 0 ¯2 0 2 1 ¯1 2 3 2 1
      ⍝⍎ nested 3 4 3⍴X2
      ⍝⍎ X3←22 24 23 24 28 29 25 23 26 28 27 25 28 25 24 23
      ⍝⍎ X3,←30 27 29 28 30 32 27 25 29 28 30 27 24 23 28 30
      ⍝⍎ X3,←25 21 24 22 27 25 26 23 27 25 26 24 24 27 28 27
      ⍝⍎ nested 3 2 4 2⍴X3
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ A←¯1+⊃⍴⍴⍵ ⋄ ⍺←A
     A←(0=≢⍺)⊃⍺ A
     TBL←A aov ⍵
     f←{U V←⍵[;0] ⋄
         FA←⊂U,1⌽')(',V~U
         FA,(+⌿0 1↓⍵)}
     K←{0=⊃0⍴⊃⍵:⍵ ⋄ ≢⍵}A
     K=2:2 3 0,⍨TBL[0;]⍪(f TBL[1 2;])⍪TBL[,3;]
     Z←TBL[0 1;]⍪(f TBL[2 5;])⍪TBL[,3;]
     Z⍪←(f TBL[4 6;])⍪TBL[,7;]
     Z,5 3 6 5 6 0
 }

 oneWay←{
          ⍝∇ One-Way Anova  (may be unbalanced)
          ⍝⍵ nested vector of vectors
          ⍝⍵ vector of namespaces
          ⍝⍵ matrix
          ⍝⍎ oneWay D.Height D.Party
          ⍝⍎ oneway D 'Height Party'
          ⍝ X←7 7 15 11 9 , 12 17 12 18 18,14 18 18 19 19,19 25 22 19 23, 7 10 11 15 11
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ ⍺←,'A'  ⍝
     W←1⊃,⍵
     9=⎕NC'W':anovaSummary ⍵  ⍝ If vector of namespaces
     A←{0=≢⍵:,'A' ⋄ ⍵}⍺
     W←↓⍣(2=⊃⍴⍴⍵)⊢⍵           ⍝ If matrix make vector of vectors
     W←{2<≢⍵:⍵
         ,0 1↓↑{⍺ ⍵}⌸/⌽⍵}W
     YN←(2*⍨+/Y)÷≢,Y←∊W
     SSA←(2*⍨+/¨W)+.÷≢¨W
     SST←Y+.*2
     SS←-⍨\SSA SST-YN
     DF←-⍨\¯1+≢¨W Y
     A'Error',SS,DF,⍪2 0
     
 }

 randomFactor←{
      ⍝⍟ Copyright (C) 2022 by Stephen M. Mansour
      ⍝∇ Generate two randomFactor model
      ⍝∊ ANOVA
      ⍝⍵ X:    Response - n-dimensional numeric vector; trailing dimension is replicates
      ⍝← TBL:  [Source|SS|DF|MS|F|P]
      ⍝⍕ NS ←  randomFactor X
      ⍝⍎ X←130 155 74 180,34 40 80 75, 20  70 82 58
      ⍝⍎ X,←150 188 159 126, 136 122 106 115, 25 70 58 45
      ⍝⍎ X,←138 110 168 160, 174 120 150 139, 96 104 82 60
      ⍝⍎ nested 3 3 4⍴X
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ A←¯1+⊃⍴⍴⍵ ⋄ ⍺←A
     A←(0=≢⍺)⊃⍺ A
     EFF SS DF←↓⍉A factor ⍵
     MS←SS÷DF
     J←2 2 3
     F←MS[⍳3]÷MS[J]
     DF12←DF[⍳3],¨DF[J]
     P←DF12 fDist prob>¨F
     EFF,SS,DF,MS,(F,0),⍪P,0
 }

 rowmajor2std←{
     ⍝ Convert row major order to standard order
     ⍝ ⍵ row major order
     ⍝← standard order
     ,⍉⍵⍴⍨(2⍟≢,⍵)/2}

 toArray←{
     ⍝Vector of equal-length vectors or NS 'Data Factor1 Factor2 ... FactorN'
     ⎕IO←0 ⋄ ⎕ML←1
     1<⍴⍴⍵:⍵  ⍝ If already matrix or higher rank, done
     L D←{1=≢⊃⍵:(' 'toNestedVector 1⊃⍵)(⍵[0]⍎1⊃⍵)
         (⎕A⍨↑≢⍵)⍵}⍵ ⍝ If 1st item namespace, unpack
     DD←↓(↑D)[;⍋⍉↑1↓D]                     ⍝ Sort data by levels
     X←⊃DD                                 ⍝ Data are first item
     1=≢D:'No Treatments Specified'        ⍝ Check for treatments
     
     A←toNestedVector¨1↓D                  ⍝ Make factor(s) into nested vector(s)
     S←(≢∘∪¨A)                                 ⍝ Find unique Values
     X⍴⍨S,(⍴X)÷×/S
     i←↓↑u⍳¨a                              ⍝ Get indexes
     b←i{⍉⍺∘.=⍳≢⍵}¨u                       ⍝ Selection matrix(ices)
     n←+/¨b                                ⍝ Determine sample size(s)
     (1∨.≠≢∘∪¨n)∧2<≢d:'Unbalanced Design'  ⍝ Don't permit unbalanced 2-way ANOVA
     
     
     
     
     
     
     
 }

 anova←{
 ⍝⍟ Copyright (C) 2018, 2023 by Stephen M. Mansour
 ⍝∇ Perform ANalysis Of VAriance (ANOVA)
 ⍝∊ Operator
 ⍝⍺ [Fixed]       Is Number of fixed factors;  Fixed∊{0,1,2,3}
 ⍝⍺ [Random←0]    Is Number of random factors  Random∊{0,1,2}
 ⍝⍺ [LS]          Cm N×N matrix
 ⍝⍺ [Alias]       Cv Source(s) of variation conflated with error.
 ⍝⍺⍺ design f       oneWay, blocked, latinSquare, factor, factorial2k, nested
 ⍝⍵ Database      W  Contains exactly one Numeric variable and one or more character variables
 ⍝⍵ Response      X  Partitioned Vector, Simple Vector, Matrix or Array, or Summary Namespace
 ⍝⍵ Levels        Cv For one-way Anova
 ⍝⍵ Factors       Cn For multi-factor Anova
 ⍝← NS:   Namespace containing the following variables:    AnovaTable  RSq  RSqAdj  S  Type
 ⍝⍕ Model ← oneWay|blocked|nested|[LS]latinSquare|[Fixed [Random]]factor|[Alias]factorial2k anova DataBase|Response Levels|Factors]]
 ⍝⍎ report oneWay anova Environment Party         ⍝ One-way ANOVA with Grouping Variable
 ⍝⍎ report oneWay anova Group1 Group2 Group3      ⍝ One-way Anova with Partioned Vector
 ⍝⍎ report N factor anova Array                   ⍝ Factorial Design
 ⍝⍎ report factorial2k anova Matrix               ⍝ 2*k factorial design
 ⍝⍎ report 'ABC' factorial2k anova Matrix         ⍝ 1/2 Fractional Factorial Design
 ⍝⍎ report 'ABC' 'DEF' factorial2k anova Matrix   ⍝ 1/4 Fractional Factorial Design
 ⍝⍎ report oneWay anova 'Height Party' selectFrom DB ⍝ Uses all variables
 ⍝⍎ report oneWay anova DB 'Environment Party'     ⍝ One-way ANOVA using factor name
 ⍝⍎ report oneWay anova stats¨(5 10 3)(5 12 2)(6 11 4)   ⍝ One-Way anova with summary data
 ⍝⍎ report 2 factor anova 2 3 ⍴1.3 2.7 3.5 1.4 3.2 3.9   ⍝ Executable Example
 ⍝⍎ anova MPG AUTO GAS                     ⍝ Two-way ANOVA
     ⎕IO←0 ⋄ ⎕ML←3 ⋄ ⍺←⍬                   ⍝ Set system variables
     NL X←{isNamespace 1⊃2↑(,⍵),0:'a'⍵
         isNamespace↑⍵:{1=≢⍵:⍵.(⊢{⍺ ⍵}⍎¨)variables ⍵
             (' 'toNestedVector 1⊃⍵)(⍵[0]⍎1⊃⍵)}⍵
         (2≤⍴⍴⍵):⍬(⊃⍵)                  ⍝ If not a vector and nested, disclose
         (,2)≡⍴⍵:((1↓⎕D)↑⍨⍴⍵)⍵          ⍝ Balanced anova
         ((1↓⎕D)↑⍨⍴⍵)⍵}⍵                ⍝ One factor unbalanced
     TBL←⍺ ⍺⍺ X ⋄ EFF←TBL[;0]
     EFF←EFF{NS←↑⍵ ⋄
         isNamespace 1⊃(,⍵),0:⍺
         2=≢⍺:⍺
         ⍝(2<≢⍵):⍺
         (9=⎕NC'NS')⍲'A'≡⍥,↑⍺:⍺
         NMS←{1=≢⍵:variables ⍵ ⋄ {1↓(⍵≠' ')⊂⍵}1⊃⍵}⍵
         S←(2=≡NMS)/'*'
         Z←{↑{⍺,'*',⍵}/NMS[⍵]}¨⎕A∘⍳¨⍺~E←⊂'Error'

         Z,(E∊⍺)/E}⍵
     NS←EFF buildAnovaTable 0 1↓TBL
     NS.DF2←{0::⍬
         ¯1↓⍵.AnovaTable[TBL[;3];2]}NS         ⍝ Denominator degrees of freedom
     NS.Type←'ANOVA'                         ⍝ Namespace type is ANOVA
    ⍝ NS.Omega2←calcw2 NS.AnovaTable
     NS←NS{0=⍺.⎕NC ⍵:⍺
         ⍺.Omega2←calcw2 ⍺⍎⍵ ⋄ ⍺}'AnovaTable'
     2≠≢EFF:NS                                ⍝ If 2 or more factors, exit
     A X←NL{2=⍴⍴⍵:(⍕¨1+⍳≢⍵)(↓⍵)               ⍝ Separate Names, Data
         isNamespace↑⍵:{b←0=⍵[0].⎕NC'name'    ⍝ If names not included
             b:(,¨(≢⍵)↑⎕A)⍵                   ⍝  Use A,B,C
             ⍵.name ⍵}⍵                       ⍝ Else use specified names
         2<≢⍵:⍺ ⍵
         ↓⍉↑{⍺ ⍵}⌸/⌽⍵}X
     H←'Group' 'N' 'Mean' 'StdDev'            ⍝ Build Means Table
     NS.Means←H⍪A,⊃(count,mean,sdev)¨X
    ⍝ NS.Omega2←{(⍵[1;1]-⍵[1;2]×⍵[2;3])÷⍵[3;1]+⍵[2;3]}NS.AnovaTable
     NS.LeftArg←⍺
     isNamespace 1⊃X:NS                       ⍝ Exit if summary data
     NS.YHAT←(≢¨X)/1↓NS.Means[;2]             ⍝ Fitted Values
     NS.E←(∊X)-NS.YHAT                        ⍝ Residuals
     NS
 }


 noncentral←{
⍝⍺ Degrees of Freedom, Non-Centrality Parameter
⍝⍺⍺ Distribution e.g. fDist
⍝⍵⍵ Relation
⍝⍵ Test statistic
⍝← P-value
     DF1 DF2 Phi←3↑⍺
     F←⍵×DF1÷DF1+Phi*2
     DF1P←(2*⍨DF1+Phi*2)÷DF1+2×Phi*2
     DF1P DF2 ⍺⍺ prob ⍵⍵ F
 }


:EndNamespace 
