﻿:Namespace Distribution
(⎕IO ⎕ML ⎕WX)←1 1 3

 beta←{
       ⍝⍟ Copyright (c) by Stephen M. Mansour
       ⍝∇ Beta Distribution; random behavior of percentages and proportions
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ A  T  Shape Parameter 1
       ⍝⍺ B  T  Shape Parameter 2
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [op] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ X   T   Probability, criticalValue or sampleSize
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← A B beta [op [rel]] X
       ⍝⍎ 0.84375 ← 2 2 beta prob < .75
       ⍝⍎ 0.2063 ← 1 3 beta critVal < 0.5
       ⍝⍎ 0.54 1.26 1.5 1.26 0.54←2 2 beta 0.1 0.3 0.5 0.7 0.9
       ⍝⍎ 1 1 1←1 1 beta 0.3 0.5 0.7  ⍝ 0 1 rectangular distribution
       ⍝⍎ 0.0005 0.0405 0.3125 1.2005 3.2805←5 1 beta 0.1 0.3 0.5 0.7 0.9
     x←⍵ ⋄ ⎕ML←3 ⋄ ⎕IO←0                   ⍝ System variables
     x≡'Type':'Continuous'                 ⍝
     (a b)c←getParameters ⍺                ⍝
     x≡'Mean':a÷a+b                        ⍝ a=Shape; b=Scale
     x≡'Variance':(a×b)÷(a+b+1)×(a+b)*2
     x≡'Mode':(a-1)÷a+b-2
     x≡'Skewness':{z←(1 2++/⍵)×.*0.5 ¯1
         ×/2(-/⍵)z,⍵*¯0.5}b a
     x≡'Kurtosis':a{z←(2*⍨⍺-⍵)×(1 2÷.×⍺+⍵+2 1)÷⍺×⍵
         6×(z-1)÷⍺+⍵+3}b
     2=≡a b:(↑,¨/⍺)∇¨⍵                     ⍝ Left argument
     str←c,'beta(⍵,⍵,⍵)'                   ⍝ Build string for R
     rAv:+##.∆r.x str ⍵ a b                ⍝ If requested use R
     0∊a b:c x{c x←⍺                       ⍝ If a=0 and/or b=0
         w←0.5-0.5×-/×⍵                    ⍝ w∊{0,0.5,1}
         c='d':(⌊/⍬)×(x∊0 1)∧w≠x           ⍝ Use R values
         c='p':x{⍺∊0 1:⍺ ⋄ ⍵}¨w            ⍝
         c='r':(?∘÷)⍣(w=0.5)⊢x⍴w           ⍝
         c='q':x{⍺∊0 1:⍺                   ⍝
             (⍺=0.5)∨⍵≠0.5:1-⍵ ⋄ ⍺>0.5}¨w  ⍝
         'Nonce Error'}a b                 ⍝
          ⍝I←a b∘{a b←⍺                         ⍝ Incomplete Beta Function
     I←{a b←⍺                              ⍝ Incomplete Beta Function
         HyperG←{a b z←⍵                   ⍝ Hypergeometric Function
             f←{⍵×(z÷⍺+1)×(×/a+⍺)÷×/b+⍺}   ⍝ Recursive Kummer terms
             0{x t←⍵ ⋄ x=x+t:x             ⍝
                 (⍺+1)∇(x+t)(⍺ f t)}0 1}   ⍝
         z←HyperG(a,1-b)(a+1)⍵             ⍝
         z×(⍵*a)÷a}                        ⍝
     Finv←{⍵≤0:0 ⋄ a b←⍺                   ⍝ Inverse cumulative dist
         P0←{m←a÷a+b                       ⍝ Estimate mean
             s←(a×b)÷(a+b+1)×(a+b)*2       ⍝ Estimate std deviation
             m+s×1 2*¯1*⍵<0.5}⍵            ⍝ Get startingpoints
         G←⍵∘{⍺-↑a b I ⍵}¨                 ⍝ Build function G(x)
         9○G SecAlg P0,G P0}               ⍝ Use secant alg to solve G(x)=0
     B←÷b×(a-1)!a+b-1                      ⍝ Beta Function
     c='d':(÷B)×{
         pr←1≠×/×⍵∘.-0 1                   ⍝ x∊[0,1]?
         pr×(⍵*a-1)×(1-⍵)*b-1}x            ⍝ Density
     c='p':(a b∘I¨0⌈1⌊x)÷B                 ⍝ Cumulative Distribution
       ⍝   c='q':9○a b Finv¨B×x                 ⍝ Inverse Distribution
     c='q':a b∘{Q←⍺ Finv B×⍵ ⋄ 0≥×/Q-0 1:Q ⍝ If outside range [0,1]
     
         1-(⌽⍺)Finv B×1-⍵}¨x               ⍝ Recalculate
     c='r':a b(16808⌶)'Beta'⍵              ⍝ Random Variable
     'Nonce Error'
 }

 binomial←{
          ⍝⍟ Copyright (c) by Stephen M. Mansour
          ⍝∇ Binomial Distribution
          ⍝∊ Distribution Function; Discrete
          ⍝⍺ N  N  Number of Trials
          ⍝⍺ P  P  Probability of success
          ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
          ⍝⍵ X   I Number of successes
          ⍝← Q   Q Probability of X successes
          ⍝⍕ Syntax:  Q ← N [P|_0.5] binomial X
          ⍝⍎ 0.27344 ← 7 0.5 binomial 3        ⍝ Probability of exactly 3 heads in 7 coint tosses
          ⍝⍎ 0.003215 ← 5 (÷6) binomial 4      ⍝ Probability of exactly 4 sixes in a toss of 5 dice.
     ⍺←1 0.5 ⋄ x←⍵ ⋄ ⎕ML←3 ⋄ ⎕IO←0        ⍝ Set system variables
     x≡'Type':'Discrete'                  ⍝
     (n p)c←getParameters ⍺               ⍝
     n p←n{⍺≠⍵:⍺ ⍵ ⋄ ⍺=⌊⍺:⍺ 0.5 ⋄ 1 ⍵}p   ⍝ Default n=1; p=0.5
          ⍝ (c≠¯2)∧~'X123'check ⍺:⎕SIGNAL 5    ⍝ Parameters must be numeric, 2 chars
          ⍝ ~'O'check c:⎕SIGNAL 11             ⍝ Operator domain error
          ⍝  ~'IU'check n:⎕SIGNAL 11           ⍝ N must be positive interger
          ⍝     ~'Q'check p:⎕SIGNAL 11         ⍝ P must be a probability
          ⍝ 1<≡n p:n ∇{⍺ ⍵ c ⍺⍺ x}¨p           ⍝ If multiple n, expand for simulation
     x≡'Mean':n×p                         ⍝
     x≡'Variance':n×p×1-p                 ⍝
     x≡'Median':y←↑{(⍵≥0.5)/⍵}n p ∇ prob≤y←(⌊,⌈)n×p
     x≡'Mode':y⊃⍨</n p ∇ y←¯1 0+⌊p×n+1
     x≡'Skewness':(1-2×p)÷(n×p×1-p)*÷2
     x≡'Kurtosis':1 0÷.+¯6 n×p×1-p
     2=≡n p:(1⌽¨c,¨n,¨p)∇¨⍵               ⍝ Left argument pervasive
     ~'X'check ⍵:⎕SIGNAL 11               ⍝ Right argument must be numeric
     str←c,'binom(⍵,⍵,⍵)'                 ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ n p               ⍝ If requested, use R
          ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     f←{(⍵!n)×(p*⍵)×(1-p)*0⌈n-⍵}          ⍝ Binomial formula
     c='d':(⍵=⌊⍵)×f ⍵                     ⍝ Probability X=x
     (c='p')∧n>200:+/¨f⍳¨⌊1+⍵             ⍝ Cumulative for Large n
     c='p':(⍵≥0)×(⊂⌊0⌈⍵)⌷0{⍺≥⌈/x:+\⍵      ⍝ Cumulative Probability X≤x
         y←(p÷1-p)×(n-⍺)÷⍺+1              ⍝ y=P(x+1)÷P(x)
         (⍺+1)∇ ⍵,⍵[⍺]×y},(1-p)*n         ⍝
     c='q':+/x∘.>0{(+/⍵)≥⌈/x:+\⍵          ⍝ Critical Value
         y←(p÷1-p)×(n-⍺)÷⍺+1              ⍝ y=P(x+1)÷P(x)
         (⍺+1)∇ ⍵,⍵[⍺]×y},(1-p)*n         ⍝
     c='r':p+.>?(n,⍵)⍴0                   ⍝ Random Variable
     'Nonce Error'
 }

 chiSquare←{
       ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ Chi-Square Distribution; Special case of Gamma Distribution
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ DF  N Degrees of Freedom
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [op] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ Value  S Test Statistic, probability or Sample Size
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← DF chiSquare [op [rel]] Value
       ⍝⍎ 11.07 ← 5 chiSquare critVal < .05 ⍝ Value from ChiSquare Table
       ⍝⍎ 0.023379 ←   5 chiSquare prob > 13  ⍝  P-Value
       ⍝⍎ 0.097304 ← 5 chiSquare 6   ⍝ Density
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ w←⍵                       ⍝ Set System variables
     w≡'Type':'Continuous'
     m c←getParameters ⍺
     w≡'Mean':m                                ⍝ Theoretical Values
     w≡'Variance':2×m                          ⍝      "        "
     w≡'Skewness':(8÷m)*÷2                     ⍝      "        "
     w≡'Kurtosis':12÷m                            ⍝      "        "
     w≡'Median':m×(1-2÷9×m)*3                     ⍝      "        "
     w≡'Mode':0⌈m-2                               ⍝      "        "
     (1≠≢m)∧c≠'r':∊(m,¨c)∇¨⍵
     str←c,'chisq(⍵,⍵)'                           ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ m                         ⍝ If requested, use R
     p←(m÷2),2                                    ⍝ Paramters for gamma distribution
     c='d':p gamma 0.001⌈w                          ⍝ Chi-Square special case of gamma
     c='p':m{⍵≤0:0 ⋄
         ⍺=1:1-2×normal prob>⍵*÷2                 ⍝ 1 df
         ##.Koko.Distrib.ChiSq_P ⍵ ⍺}¨⍵  ⍝
     c='q':m{##.Koko.Distrib.ChiSq_Xa ⍵ ⍺}¨1-⍵
     ⍝ c=1:p{m>250:normal prob<(⍵-m)÷(2×m)*÷2
     ⍝         si←(⍺≡⌊⍺)∨15≥↑⍺                       ⍝ If small and integer
     ⍝         si:⍺ gamma prob<0⌈⍵                   ⍝ Use gamma
     ⍝         x←{⌊⍵+¯1 0 1}↑⍺                       ⍝ Otherwise
     ⍝         q←(x,¨2)gamma prob<¨⊂⍵                ⍝ find surrounding integers
     ⍝         z←⍺[0]{(⊃x ⍵)interpolate ⍺}¨↓⍉⊃q      ⍝ Interpolate for odd df
     ⍝         z}w⌊100⌈m+6×(2×m)*÷2                  ⍝
     ⍝     c=¯1:m{⍺≥69:⍺{z←0 1 ¯1 normal ⍵           ⍝ Inverse Chi-Square
     ⍝             0.5×(z+(¯1+2×⍺)*÷2)*2}w           ⍝
     ⍝         (⍺<33)∨0=2|⍺:(p,¯1)gamma ⍵            ⍝
     ⍝         x←{⌊⍵+¯1 0 1}↑p                       ⍝ find surrounding integers
     ⍝         q←(x,¨2)gamma criticalValue>¨⊂⍵       ⍝ Otherwise
     ⍝         p[0]{(⊃x ⍵)interpolate ⍺}¨↓⍉⊃q}⍵      ⍝
     1<≢m:m randomVariable¨⍵
     d←normal randomVariable m×w                ⍝ Chi-Square is sum of
          ⍝+⌿m w⍴d*2                                 ⍝ squared normal random variables
     +⌿(m,w)⍴d*2
 }

 dirac←{
      ⍝ Dirac Delta function
      ⍝⍺ value (default 0)
      ⍝⍺⍺ probability, inverse, random variable
      ⍝⍵ A partular value of the exponential random variable
     ⎕ML←3 ⋄ ⎕IO←0
     ⍺←0
     ⍵≡'Type':''
     a c←2↑⍺                         ⍝ Parameter, Condition
     ⍵≡'Mean':a
     ⍵≡'Variance':0
     ⍵≡'Median':a
     ⍵≡'Mode':a
     ⍵≡'Skewness':0
     ⍵≡'Kurtosis':0J1
     c=0:a{⌊/(⍵≠⍺)/0}¨⍵              ⍝ Density
     c=1:0.5×1+×⍵-a                  ⍝ Distribution
     c=¯1:⍵-a                        ⍝ Inverse
     ⍵⍴a                             ⍝ Random Variable
 }

 exponential←{
       ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ Exponential Distribution; Models lifetimes or distance between events in a Poisson process.
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ Lambda  T  Rate, or inverse scale
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍵ Time      T  Lifetime or Interarrival Times
       ⍝⍥ Op        oD
       ⍝⍵⍵ Rel      fR
       ⍝← Result    S  Density, Probability or Critical Value
       ⍝⍕ Result ← Lambda exponential [op [rel]] Time
       ⍝⍎ 0.03894 0.030327 0.018394 0.011157←.05 exponential 5 10 20 30 ⍝ PDF
       ⍝⍎ 0.2212 0.39347 0.63212 0.77687←.05 exponential prob ≤ 5 10 20 30 ⍝ CDF
     ⎕ML←3 ⋄ ⎕IO←0                        ⍝ Set system variables
     ⍵≡'Type':'Continuous'                ⍝
     ⍵≡'Parameters':{⍵.f←÷∘mean ⋄ ⍵}⎕NS'' ⍝
     m c←getParameters ⍺                  ⍝
     ⍵≡'Mean':÷m                          ⍝ Theoretical values
     ⍵≡'Variance':÷m*2                    ⍝       "       "
     ⍵≡'Median':(⍟2)÷m                    ⍝       "       "
     ⍵≡'Mode':0×m                         ⍝       "       "
     ⍵≡'Skewness':2*×m                    ⍝       "       "
     ⍵≡'Kurtosis':6*×m                    ⍝       "       "
     1≠≢m:∊⍣(c≠'r')⊢(m,¨c)∇¨⍵             ⍝ Make pervasive
     str←c,'exp(⍵,⍵)'                     ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ m                 ⍝ If requested, use R
     c='d':(⍵≥0)×m×*-⍵×m                  ⍝ Density
     c='p':1-*-m×⍵⌈0                      ⍝ Distribution
     c='q':-(⍟1-⍵)÷m                      ⍝ Inverse
     c='r':2-/+\(÷m)×⍟?0⍴⍨⍵+1             ⍝ Random Variable
     'Nonce Error'
 }

 fDist←{                      ⍝ F  Distribution
       ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ Density of F Distribution; Special case of Gamma Distribution
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ DF1     I Numerator Degrees of Freedom   (0=Infinite)
       ⍝⍺ DF2     I Denominator Degrees of Freedom (0=Infinite)
       ⍝⍺ [Phi←0] S Non-Central Parameter
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [operator] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ Value  S Test Statistic, probability or Sample Size
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← DF1 DF2 [Phi|_0] fDist [operator [rel]] Value
       ⍝⍎ 3.4903 ← 3 12 fDist critVal < .05 ⍝ Value from F-Table
       ⍝⍎ 0.3459 ←   3 12 fDist prob > 4  ⍝  P-Value
       ⍝⍎ 0.17229 ← 5 3 fDist 1.8   ⍝ Density
     w←⍵ ⋄ ⎕ML←3 ⋄ ⎕IO←0 ⋄ ⎕FR←1287 ⋄ ⎕DIV←0        ⍝ Set system variables
     w≡'Type':'Continuous'                    ⍝
     df c←getParameters ⍺
          ⍝Phi2 c←{2=≢⍵:⍵*2 1 ⋄ 0=≢⍵:0 ⋄ ⍵∊0 1 ¯1 ¯2:0 ⍵ ⋄ ⍵ 0}2↓⍺
     w≡'Mean':⎕NULL@{2≥1⊃df}{⍵÷1⌈⍵-2}1⊃df
     w≡'Variance':⎕NULL@{4≥1⊃df}↑{2×(⍵*2)×(⍺+⍵-2)÷⍺×(2*⍨⍵-2)×1⌈⍵-4}/df
     w≡'Mode':(df+¯2 2)×.÷df
     w≡'Skewness':⎕NULL@{6≥1⊃df}↑{(¯2+⍵+2×⍺)×((8×⍵-4)*÷2)÷(⍵-6)×(⍺×⍺+⍵-2)*÷2}/1 7⌈df
     w≡'Kurtosis':⎕NULL
     2=≡df:⍵ ∇⍨¨↑c,¨⍨¨,¨/(⌈/≢¨df)⍴¨df           ⍝ Left argument pervasive
     Phi2←×⍨↑⌽3↑df,0
     str←c,'f(⍵,⍵,⍵,⍵)'                         ⍝ Build R expression
     rAv:+##.∆r.x str ⍵,df,Phi2                 ⍝ Use R if requested
     R←df[0]÷df[0]+Phi2
     df[0]←{⍵=0:⍵ ⋄ (2*⍨⍵+Phi2)÷⍵+2×Phi2}df[0]  ⍝ Adjust numerator df?
     df←2↑df                                    ⍝ Remove Noncentral parameter
     FDEN←{MN←254⌊⍺+(172<⍺)×2|⍺                 ⍝ Density
         M←↑MN ⋄ H←÷2                           ⍝ (Adjust DF to prevent domain error -
         Z←H×M×!/¯1+H×+\⌽MN                     ⍝ may cause slight inaccuracy)
         Z←Z×÷/MN*M×H
         Z←Z×⍵*H×M-2
         Z÷(1+⍵×÷/MN)*H×+/MN}
     FApprox←{0∧.=⍺:1  ⍝ Abramowitz & Stegun p. 947 Eq. 26.6.11 & 26.6.12
         0=↑⍺:⍺[1]÷⍺[1]chiSquare criticalValue>⍵
         (⍺[0]chiSquare criticalValue<⍵)÷⍺[0]}
     c='d':df FDEN 0.001⌈w                       ⍝ Density
     c='p':{
         df[0]=0:df[1]chiSquare prob>df[1]÷⍵
         df[1]=0:df[0]chiSquare prob<⍵×df[0]
         0⌈##.Koko.Distrib.Fratio_P ⍵,df}¨w×R
     c='q':R÷⍨{0∊df:df FApprox ⍵ ⋄
         ##.Koko.Distrib.Fratio_Fa ⍵,df}¨1-w
     X←df chiSquare randomVariable¨w          ⍝ F Distribution is ratio of
     ↑÷/X÷df                                  ⍝ two chi-Square distributions
 }

∇ Z←gamain Y;⎕IO;PN;ACU;DIF;OFLO;GIN;IFAULT;P;G;I;X;TERM;RN;FACTOR;A;B;AN
 ⎕IO←1
     ⍝ X P G IFAULT←Y
 X P←Y
 G←⍟!¯1+P
 PN←6⍴0
 ACU←1E¯8
 OFLO←1000000000000000000000000000000
 GIN←0
 IFAULT←0
 :If P≤0 ⋄ IFAULT←1 ⋄ :EndIf
 :If X<0 ⋄ IFAULT←2 ⋄ :EndIf
 →((IFAULT>0)∨X=0)/L50
 FACTOR←*(P×⍟X)-X+G
 →((X>1)∧(X>P))/L30
 GIN←1
 TERM←1
 RN←P
L20:RN←RN+1
 TERM←TERM×X÷RN
 GIN←GIN+TERM
 →(TERM>ACU)/L20
 GIN←GIN×FACTOR÷P
 →L50
L30:A←1-P
 B←A+X+1
 TERM←0
 PN[⍳4]←1 X(X+1)(X×B)
 GIN←÷/PN[3 4]
L32:A+←1
 B+←2
 TERM+←1
 AN←A×TERM
 :For I :In 1 2
     PN[I+4]←(B×PN[I+2])-AN×PN[I]
 :EndFor
 →(PN[6]=0)/L35
 RN←÷/PN[5 6]
 DIF←|GIN-RN
 →(DIF>ACU)/L34
 →(DIF≤ACU×RN)/L42
L34:GIN←RN
L35:
 :For I :In ⍳4
     PN[I]←PN[I+2]
 :EndFor
 →(|PN[5]<OFLO)/L32
 :For I :In ⍳4
     PN[I]←PN[I]÷OFLO
 :EndFor
 →L32
L42:GIN←1-FACTOR×GIN
L50:Z←GIN
∇

 gamma←{                ⍝ Gamma Distribution
       ⍝⍟ Copyright (c) by Stephen M. Mansour
       ⍝∇ Gamma Distribution
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ K      T  Shape Parameter 1
       ⍝⍺ Theta  T  Scale Parameter 2
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [op] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ X        T   Probability, CriticalValue or sampleSize
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← K Theta beta [op [rel]] X
       ⍝⍎ 0.065642 ← 4 2 gamma prob < 3   ⍝ Cumulative Distribution Function
       ⍝⍎ 0.062755 ← 4 2 gamma 3          ⍝ Density
         ⍝ 11::0
     x←⍵ ⋄ ⎕ML←3 ⋄ ⎕IO←0              ⍝ Set system variables
     x≡'Type':'Continuous'
     (k r)c←getParameters ⍺           ⍝ Get parameters
     x≡'Mean':k×r                     ⍝ Calculate theoretical values
     x≡'Variance':k*r*2               ⍝      "         "         "
     x≡'Median':k r ∇ critVal<0.5     ⍝      "         "         "
     x≡'Mode':r×k-1                   ⍝      "         "         "
     x≡'Skewness':2÷k*÷2              ⍝      "         "         "
     x≡'Kurtosis':6÷k                 ⍝      "         "         "
     2=≡k r:(1⌽¨↑,¨/c k r)∇¨⍵         ⍝ Left argument pervasive
     str←c,'gamma(⍵,⍵,⍵)'             ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ k r           ⍝ Use R if requested
     F←k r∘{                          ⍝ Cumulative distribution
         k r←⍺ ⋄ c←⍵÷r
         g←{x←⍵
             s←0{y←*(⍺×⍟x⌈⎕CT)-⍟!k+⍺
                 (170≤k+⍺)∨(⍺>10)∧⎕CT>|y:⍵
                 (⍺+1)∇ ⍵+y}0
             s×*(⍺×⍟⍵⌈⎕CT)+(⍟!⍺-1)-⍵}
         k≠⌊k:|(k g¨c)÷!k-1
         i←⍳k
         |1-+/*(-⍟!i)+(i×⍟⎕CT⌈c)-c}
     Finv←{⍵≤0:0
         P0←k r{m←(×/⍺)-↑⍺                  ⍝ Estimate mean
             s←⍺×.*0.5 1                    ⍝ Estimate std deviation
             (m>2)∨⍵>0.5:m+s×1 2*¯1*⍵<0.5
             m=1.5:0.05 1.5
             m<2:0.0002 0.0001
             (m=2)∧⍵<0.1:0.2 0.8
             (m=2.5)∧⍵<0.025:0.4 0.6
             m+0 1}⍵
         G←⍵∘{⍺-↑F ⍵}¨
         G SecAlg P0,G P0}
     c='d':(r*-k)×(x*k-1)×(*-x÷r)÷!k-1        ⍝ Density
     c='p':F¨x                                ⍝ Distribution
     c='q':Finv¨x                             ⍝ Inverse
     k r(16808⌶)'Gamma'⍵                      ⍝ Random Variable
 }

 geometric←{
          ⍝⍟ Copyright (c) by Stephen M. Mansour
          ⍝∇ Geometric Distribution
          ⍝∊ Distribution Function; Discrete
          ⍝⍺ P  P  Probability of success in one trial
          ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
          ⍝⍵ N   I Number of Trials
          ⍝← Q   Q Probability of at least one success in N trials
          ⍝⍕ Syntax:  Q ← P geometric N
          ⍝⍎ 0.25 ←  0.5 geometric 2
     x←⍵ ⋄ ⎕ML←3 ⋄ ⎕IO←0
     x≡'Type':'Discrete'
     3=⎕NC'⍺':∆SigErr 14 'Left argument missing'
     p c←getParameters ⍺
     ⍵≡'Mean':÷p
     ⍵≡'Mode':1
     ⍵≡'Median':⌈-÷2⍟1-p
     ⍵≡'Variance':(1-p)÷p*2
     ⍵≡'Skewness':(2-p)÷(1-p)*÷2
     ⍵≡'Kurtosis':6+(p*2)÷1-p
         ⍝str←'pdqr'[1 0 ¯1 ¯2⍳c],'geom(⍵,⍵)' ⍝ Build R expression
     1<≢p:(p,¨c)∇¨⍵
     str←c,'geom(⍵,⍵)' ⍝ Build R expression
          ⍝rAv:+##.∆r.x str(⍵-1)p                ⍝ Use R if requested
     rAv:+##.∆r.x str(⍵-c∊'dp')p          ⍝ Use R if requested
     c='d':((x>0)∧x=⌈x)×p×(1-p)*x-1        ⍝ Prob X=x
     c='p':1-(1-p)*⌊x⌈0                    ⍝ Prob X≤x
     c='q':⌊(1-p)⍟1-x                      ⍝ Inverse prob
     c='r':1+⌊(1-p)⍟?⍵⍴0                   ⍝ Random Variable
     'Nonce Error'
 }

 hyperGeometric←{                                ⍝ Scalar,Arithmetic
          ⍝⍟ Copyright (c) by Stephen M. Mansour
          ⍝∇ Hypergeometric Distribution
          ⍝∊ Distribution Function; Discrete
          ⍝⍺ K I  Successes in population
          ⍝⍺ M M  Sample size
          ⍝⍺ N N  Population size
          ⍝⍵ X I  Number of successes in sample
          ⍝← P P  Probability of X successes
          ⍝⍕ Syntax:  K M N hyperGeometric X
          ⍝⍎ 0.03993 ← 4 5 52 hyperGeometric 2 ⍝ Probability of 2 aces in a poker hand
          ⍝⍎ 0.12469 ← 13 13 52 hyperGeometric 5 ⍝ Probability of 5 card spade suit in a bridge hand
     'Type'≡x←⍵:'Discrete'
         ⍝ m n N c←4↑⍺ ⋄ ⎕ML←3 ⋄ ⎕IO←0             ⍝ Set system variables
     ⎕ML←3 ⋄ ⎕IO←0
     (m n N)c←getParameters ⍺
        ⍝  m n N∨.<0 1 1:∆SigErr'All parameters must be positive'
     ∨/∊m n N<0 1 1:∆SigErr'All parameters must be positive'
     x≡'Mean':m×n÷N                           ⍝ Theoretical Values
     x≡'Median':m n N ∇ critVal<0.5           ⍝    "           "
     x≡'Variance':N{×/(⍵,⍺-⍵),÷⍺-3↑1}m n      ⍝    "           "
     x≡'Mode':⌊(m n×.+1)÷N+2                  ⍝    "           "
     x≡'Skewness':N{z←(⍺-1)÷×/⍵,⍺-⍵           ⍝    "           "
         (z*÷2)×(×/⍺-2×⍵)÷⍺-2}m n             ⍝
     x≡'Kurtosis':N{u←×/⍵,⍺-⍵                 ⍝    "           "
         z←(⍺-1)×(⍺*2)×(⍺×⍺+1)-6+.×⍵×⍺-⍵      ⍝
         z+←6×u×⍺⊥5 ¯6                        ⍝
         z÷×/u,⍺-2 3}m n                      ⍝
         ⍝ str←'pdqr'[1 0 ¯1 ¯2⍳c],'hyper(⍵,⍵,⍵,⍵)' ⍝ Build R Expression
     str←c,'hyper(⍵,⍵,⍵,⍵)' ⍝ Build R Expression
     rAv:+##.∆r.x str ⍵ m(N-m)n               ⍝ Use R if requested
     1≠≡m n N:(↑,¨/m n N c)∇¨⍵
          ⍝c='d':{p←((⍵!m)×(n-⍵)!N-m)÷n!N             ⍝ Probability (X=x)
     c='d':{p←{↑(!/-⍨/⍵)×÷/!⌿⍵}⍵ m,⍪n N
         p×(⍵=⌈⍵)∧⍵≥0}⍵                       ⍝
     c='p':(x≥0)×(⊂⌊0⌈x)⌷1{⍺>⌈/x:+\⍵            ⍝ Cumulative Probability X≤x
         11::+\m n N hyperGeometric⍳1+⌈/x     ⍝ Use brute force if ÷0
         y←(m n×.-⍺-1)÷⍺×⍺+N-m+n              ⍝ y=P(x+1)÷P(x)
         (⍺+1)∇ ⍵,⍵[⍺-1]×y},÷/n!⌽-\N m        ⍝
         ⍝c='q':+/x∘.≥1{(+/⍵)>⌈/x:+\⍵             ⍝ Critical Value???
     c='q':1+x⍸⍨1{(+/⍵)>⌈/x:+\⍵               ⍝ Critical Value???
         11::+\m n N hyperGeometric⍳m⌊n     ⍝ Use brute force if ÷0
         y←(m n×.-⍺-1)÷⍺×⍺+N-m+n              ⍝ y=P(x+1)÷P(x)
              ⍝  11::+\m n N hyperGeometric⍳m⌊n    ⍝ Use brute force if ÷0
         (⍺+1)∇ ⍵,⍵[⍺-1]×y},÷/n!⌽-\N m        ⍝
     c='r':{m+.>n?N}¨⍳⍵                       ⍝ Random Variables
     'Nonce Error'
 }

 kolmogorov←{                                ⍝ Scalar,Arithmetic
          ⍝⍟ Copyright (c) 2022 by Stephen M. Mansour
          ⍝∇ Kolmogorov Distribution
          ⍝
          ⍝∊ Discrete Distribution Function
          ⍝⍺ N - Sample Size
          ⍝⍵ X - Value of statistic
          ⍝← Density?
          ⍝⍕ Syntax:  N kolmogorov X
          ⍝⍎ 0.975 ← 10 kolmogorov prob ≤ .409
     'Type'≡x←⍵:'Continuous'
     ⎕ML←3 ⋄ ⎕IO←0                   ⍝ Set system variable
     N C←getParameters ⍺
         ⍝ N∨.<0 1:∆SigErr'All parameters must be positive'
     g←{T←⍵ ⋄ N←⍺ ⋄ J←⍳⌊1+N×1-T
         1-T×+/{P←T+⍵÷N ⋄ P÷⍨N P binomial prob=⍵}¨J}
         ⍝  x≡'Mean':M×N÷2                              ⍝ Theoretical Values
     ⍝     x≡'Variance':M×N×(M+N+1)÷12                 ⍝    "           "
       ⍝   ∨/~M N∊1+⍳10:∆SigErr'Domain Error'         ⍝ M,N integers between 1 and 10 inclusive
         ⍝ x≡'Mode':mode RF                            ⍝    "           "
     ⍝     x≡'Skewness':skewness RF                    ⍝    "           "
     ⍝     x≡'Kurtosis':kurtosis RF                    ⍝    "           "
     ⍝     str←'pdqr'[1 0 ¯1 ¯2⍳C],'wilcox(⍵,⍵,⍵)'     ⍝ Build R Expression
     ⍝    rAv:+##.∆r.x str ⍵ M N                      ⍝ Use R if requested
     1<≢⍵:⍺∘∇¨⍵                                  ⍝ Allow array arguments
     C='d':(-/N g¨⍵(+,-)h)÷2×h←0.00001             ⍝ Probability (X=x)
     C='p':N g ⍵                                   ⍝ Cumulative Probability X≤x
     C='q':(⍵-N∘g)SecAlg D,⍵-N g¨D←1 0.7÷sqrt N   ⍝ Critical Values (Inverse)
     C='r':'Nonce Error'                          ⍝ Random Variables
 }

 logNormal←{                                ⍝ Scalar,Arithmetic⍝⍝⍝⍝⍝⍝
      ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ Lognormal Distribution; Reliability Model
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ T50   T  Median
       ⍝⍺ Sigma S  Shape Parameter
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍵ Time      T  Lifetime or Interarrival Times
       ⍝⍥ Op        oD
       ⍝⍵⍵ Rel      fR
       ⍝← Result    S  Density, Probability or Critical Value
       ⍝⍕ Result ← T50 Sigma logNormal [op [rel]] Time
       ⍝⍎ 0.010442 0.0030712 0.0015958 0.0007678←100 2.5 logNormal 10 50 100 200 ⍝ PDF f(t)
       ⍝⍎ 0.17852 0.39079 0.5 0.60921←100 2.5 logNormal prob ≤ 10 50 100 200 ⍝ CDF; F(t)
       ⍝⍎ 0.82148 0.60921 0.5 0.39079←100 2.5 logNormal prob > 10 50 100 200 ⍝ Reliability Function R(t) = 1-F(t)
     ⍵≡'Type':'Continuous'
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ ⎕DIV←1                  ⍝ Set system variables
     (T50 s)c←getParameters ⍺
     ⍵≡'Mean':T50×*0.5×s*2                   ⍝ Theoretical values
     ⍵≡'Variance':T50×*(s*2)×¯1+*s*2
     ⍵≡'Median':T50
     ⍵≡'Mode':T50÷*s*2
     ⍵≡'Skewness':{(⍵+2)×(⍵-1)*÷2}*s*2
     ⍵≡'Kurtosis':¯6+1 2 3+.×*4 3 2×s*2
         ⍝str←'pdqr'[1 0 ¯1 ¯2⍳c],'lnorm(⍵,⍵,⍵)'
     str←c,'lnorm(⍵,⍵,⍵)'
     rAv:+##.∆r.x str ⍵(⍟T50)s               ⍝
     U←s÷⍨⍟(⍵÷T50)⌈÷⌊/⍳0                     ⍝ ln(t/T50)÷s
     c='d':(*-0.5×U*2)÷s×⍵×sqrt○2            ⍝ Density
     c='p':normal prob<U                     ⍝ Cumulative Distribution
     c='q':T50×*s×normal critVal>⍵           ⍝ Inverse
     c='r':T50×*s×normal randomVariable ⍵    ⍝ Random variable
     'Nonce Error'
 }

 mannWhitneyU←{                                ⍝ Scalar,Arithmetic
          ⍝⍟ Copyright (c) 2021 by Stephen M. Mansour
          ⍝∇ Mann-Whitney U distribution
          ⍝  Annals of Mathematical Statistics 1943, Vol114. No. 1
          ⍝∊ Discrete Distribution Function
          ⍝⍺ M - number of items in Category A
          ⍝⍺ N - number of items in Category B
          ⍝⍵ R - Value of statistic
          ⍝← Probability of X runs
          ⍝⍕ Syntax:  M N mannWhitneyU R
          ⍝⍎ 0.0119 ← 3 6 mannWhitneyU 0
     'Type'≡x←⍵:'Discrete'
     ⎕ML←3 ⋄ ⎕IO←0                               ⍝ Set system variables
     (M N)C←getParameters ⍺
     1<≡M N:(C,¨⍨M,¨N)∇¨⊂⍵
         ⍝ M N∨.<0 1:∆SigErr'All parameters must be positive'
     ∨/∊M N<0 1:∆SigErr'All parameters must be positive'
     f←{K←⌈⍵÷2 ⋄ ~2|⍵:×/2,(K-1)!⍺-1              ⍝ Even Combinations
         +/×⌿0 1⌽(K-1 2)∘.!⍺-1}                  ⍝ Odd Combinations
     x≡'Mean':M×N÷2                              ⍝ Theoretical Values
     x≡'Variance':M×N×(M+N+1)÷12                 ⍝    "           "
       ⍝   ∨/~M N∊1+⍳10:∆SigErr'Domain Error'         ⍝ M,N integers between 1 and 10 inclusive
     RF←1 frequency(+/M cmat M+N)-0.5×M×M+1      ⍝ Create frequency table
     x≡'Mode':mode RF                            ⍝    "           "
     x≡'Skewness':skewness RF                    ⍝    "           "
     x≡'Kurtosis':kurtosis RF                    ⍝    "           "
         ⍝str←'pdqr'[1 0 ¯1 ¯2⍳C],'wilcox(⍵,⍵,⍵)'     ⍝ Build R Expression
     str←C,'wilcox(⍵,⍵,⍵)'                       ⍝ Build R Expression
     rAv:+##.∆r.x str ⍵ M N                      ⍝ Use R if requested
     1<≢⍵:⍺∘∇¨⍵                                  ⍝ Allow array arguments
     C='d':RF prob=⍵                               ⍝ Probability (X=x)
     C='p':RF prob≤⍵                               ⍝ Cumulative Probability X≤x
     C='q':RF criticalValue≥⍵                     ⍝ Critical Values (Inverse)
     C='r':(-0.5×M×M+1)++/⊃1+M?¨⍵/M+N             ⍝ Random Variables
 }

 multinomial←{                                 ⍝ Custom Distribution
          ⍝⍟ Copyright (c) by Stephen M. Mansour
          ⍝∇ Multinomial/Custom Distribution
          ⍝∊ Discrete Distribution Function
          ⍝⍺ Possible Values (Numeric or Character)
          ⍝⍺ Probabilities
          ⍝⍵ x - Particular Value (Numeric or Character)
          ⍝← Probability of Value
          ⍝⍕ Syntax:  N V multinomial M
          ⍝⍎ (0 1 2 3)(5 2 1 3)  multinomial 3
     ⎕ML←3 ⋄ ⎕IO←0                         ⍝ Set system variables
     ⍵≡'Type':'Discrete'                   ⍝ Discrete Distribution
     w←toNestedVector ⍵                    ⍝ If Character, make nested vector
     pv←{(1<≡⍵):0 ⋄ ∧/(⍵≥0),(⍵≤1),1≥+/⍵}   ⍝ Is this a probability vector?
     v p c←{pv ⍵:(⍳⍴,⍵)(,⍵)0               ⍝ If a simple vector assume probabilities
         b←(c←¯1↑⍵)∊¯1 0 1 ¯2              ⍝ If last item is a flag
         b∧pv ¯1↓⍵:(⍳¯1+≡⍵)(¯1↓⍵)c
         ∧/(pv,w←¯1↓⍵),b:(⍳⍴w)w c          ⍝
         3=⍴,⍵:⍵                           ⍝ If 3 items, variables, proportions, flag
         (|⊂↑⌽⍵)∊¯1+⍳4:(⊂⍳⍴,⍵),⍵           ⍝ If last item a flag, generate values
         ⍵,0}⍺                             ⍝ Else assume density
         ⍝ ~pv p:'Not a probability vector'      ⍝ Check probability vector
     p∨.<0:'Negative values not permitted'
     p←p÷+/p                               ⍝ If percents, weights or counts, convert to probability
     v p←v{0≠↑∊0⍴⍺:⍺ ⍵ ⋄ (⊂⍋⍺)∘⌷¨⍺ ⍵}p     ⍝ Sort if numeric
     v←toNestedVector v
     ⍵≡'Mean':v+.×p                        ⍝ If numeric, calculate mean
     ⍵≡'Variance':v{0≠↑0⍴↑⍺:1-+/2*⍨(⊢÷+/)⍵ ⍝ If char use Kader & Perry 2007
         (⍵+.×⍺*2)-(⍺+.×⍵)*2}p             ⍝ Else calc pop variance
     ⍵≡'Median':v[0.5+.>+\p]
     ⍵≡'Mode':↑v[p⍳⌈/p]
     ⍵≡'Skewness':skewness(⌊0.5+p×1000)/v
     ⍵≡'Kurtosis':kurtosis(⌊0.5+p×1000)/v
     c=0:(p,1-+/p)[v⍳w]                    ⍝ Get probability
     c=1:{0≠↑0⍴v:'Cumulative probability not defined on nominal data.'
         p+.×v∘.≤⍵}w
     c=¯1:{0≠↑0⍴v:'Critical value not defined on nominal data.'
         v[+/⍵∘.≥+\p]}w
     u←?⍵⍴0                                ⍝ random Variable
     v{1=⍴⍴⍺:⍺[⍵] ⋄ ⍺[⍵;]}+/u∘.≥+\p        ⍝
 }

 negativeBinomial←{
          ⍝⍟ Copyright (c) by Stephen M. Mansour
          ⍝∇ Negative Binomial Distribution
          ⍝∊ Distribution Function; Discrete
          ⍝⍺ K   N  Number of Successes
          ⍝⍺ Q   P  Probability of success in one trial
          ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
          ⍝⍵ N   N Number of Trials
          ⍝← P   P Probability of K success in N trials
          ⍝⍕ Syntax:  P ← K Q  negativeBinomial N
          ⍝⍎ 0.3125 ← 5 0.5 negativeBinomial 3      ⍝'Nonce Error'
     ⎕ML←3 ⋄ ⎕IO←0
     x←⍵
     x≡'Type':'Discrete'
     3=⎕NC'⍺':∆SigErr 14 'Left argument missing'
          ⍝r p c←3↑⍺
     (r p)c←getParameters ⍺
          ⍝2≠⍴,a.Value:∆SigErr 14 'Left argument must consist of two values'
     ⍵≡'Mean':r×(1-p)÷p
     ⍵≡'Median':r p ∇ critVal≤0.5
     ⍵≡'Mode':(r>1)×⌊(1-p)×(r-1)÷p
     ⍵≡'Variance':r×(1-p)÷p*2
     ⍵≡'Skewness':(2-p)÷(r×1-p)
     ⍵≡'Kurtosis':(6÷r)+(p*2)÷r×1-p
         ⍝ str←'pdqr'[1 0 ¯1 ¯2⍳c],'nbinom(⍵,⍵,⍵)' ⍝ Build R expression
     str←c,'nbinom(⍵,⍵,⍵)'                   ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ r p                  ⍝ If requested, use R
     c='d':{p←(⍵!r+⍵-1)×(p*r)×(1-p)*⍵          ⍝ Probability mass function
         p×(⍵=⌈⍵)∧⍵≥0}⍵                      ⍝
     c='p':(x≥0)×(⊂⌊0⌈x)⌷0{⍺≥⌈/x:+\⍵           ⍝ Cumulative Probability X≤x
         y←(1-p)×(r+⍺)÷⍺+1                   ⍝ y=P(x+1)÷P(x)
         (⍺+1)∇ ⍵,⍵[⍺]×y},p*r                ⍝
     c='q':+/x∘.>0{(+/⍵)≥⌈/x:+\⍵              ⍝ Inverse
         y←(1-p)×(r+⍺)÷⍺+1
         (⍺+1)∇ ⍵,⍵[⍺]×y},p*r
     c='r':+⌿⌊(1-p)⍟?r ⍵⍴0                    ⍝ Random variable
     c≠'r':'Nonce Error'
     m←⌈x×(1,2÷x*÷2)+.×r×(1-p)÷p*1 2         ⍝ Sample Size
         ⍝ b←p>(0.5+?m⍴k)÷k←¯1+2*31               ⍝ successes
     b←p>?m⍴0
     i←(+\b)⍳r×1+⍳x                          ⍝ indexes
     ¯2-/0,+/¨i↑¨⊂~b
 }

 normal←{                                ⍝ Scalar,Arithmetic⍝⍝⍝⍝⍝⍝
       ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ (Standard) Normal Distribution
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ [Mu←0]    X Population mean
       ⍝⍺ [Sigma←1] S Population standard deviation
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [op] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ Value    S Test Statistic, probability or Sample Size
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← [Mu|_0 Sigma|_1] normal [op [rel]] Value
       ⍝⍎ 0.975 ← normal prob ≤ 1.96      ⍝ Probability from Standard Normal Distribution Table; p-Value
       ⍝⍎ 1.645 ← normal critVal ≤ .05    ⍝ Standard Critical Value
       ⍝⍎ 0.25249 ← 68 3 normal prob ≥ 65 ⍝ Probability that someone is taller than 65 inches
       ⍝⍎ 72.935 ← 68 3 normal critVal ≥ .95 ⍝ 95th percentile of height
     ⍺←0 1 ⋄ ⎕ML←3 ⋄ ⎕IO←0                  ⍝ Default to standard normal
     ⍵≡'Type':'Continuous'                  ⍝ Continuous Distribution
     ⍵≡'Parameters':{⍵.f←(mean,sdev) ⋄ ⍵}⎕NS'' ⍝ Get parameter estimates
          ⍝m s c←3↑⍺                             ⍝ Mean, sdev, action
     (m s)c←getParameters ⍺
       ⍝   m s←⍺ ⋄  ∆∆←0 ⋄ c←∆∆
     'Mean' 'Median' 'Mode'∊⍨⊂⍵:m           ⍝ Obtain theoretical values
     'Variance'≡⍵:s*2                       ⍝
     'Skewness' 'Kurtosis'∊⍨⊂⍵:0            ⍝     "    "          "
        ⍝ 2=≡⍺:(↑,¨/⍺)∇¨⍵                        ⍝ PervasiveLeft argument
        ⍝  2=≡m s:(↑,¨/⍺)∇¨⍵                        ⍝ PervasiveLeft argument
     2=≡m s:(1⌽¨c,¨m,¨s)∇¨⍵
         ⍝ ~'X23'check ⍺:⎕SIGNAL 5                ⍝ Length error
     ⍝     ~'S'check s:⎕SIGNAL 11                 ⍝ Sdev must be > 0
     ⍝     ~'O'check c:⎕SIGNAL 11                 ⍝ Check Operator
     ⍝     ~'X'check ⍵:⎕SIGNAL 11                 ⍝ Right argument must be numeric
         ⍝str←'pdqr '[1 0 ¯1 ¯2⍳c],'norm(⍵,⍵,⍵)' ⍝ Build R expression
     str←c,'norm(⍵,⍵,⍵)' ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ m s                 ⍝ Use R if requested
     sden←{m s←⍺                            ⍝ Normal Density
         (÷s×(○2)*÷2)×*-0.5×(s÷⍨⍵-m)*2}     ⍝  e ^ (x^2)- 2
     snorm←{d←0.3193815 ¯0.3565638 1.781478 ¯1.821256 1.330274  ⍝ Cumulative Dist
         |(⍵≥0)-(0.3989423×*¯0.5×⍵*2)×d+.××⍀(5,⍴⍵)⍴÷1+0.2316419×100⌊|⍵}
     Inv←{⍺←1 2                             ⍝ Inverse operator
         f←⍺⍺{⍺-⍺⍺ ⍵}                       ⍝ Solve f(x)=0 for x
         ⍵∘f{⍺⍺ SecAlg ⍵,⍺⍺ ⍵}⍺}            ⍝ Use secant algorithm
     Rand←{u0←?⍵⍴0 ⋄ u1←?⍵⍴0                ⍝ Generate two uniform random vars
         ((-2×⍟u0)*÷2)×2○○2×u1}             ⍝ Transform and combine them
     c='d':m s sden ⍵                         ⍝ Density
     c='p':snorm(⍵-m)÷s                       ⍝ Cumulative Distribution
     c='m':*(m×⍵)+0.5×(s×⍵)*2                ⍝ Moment Generating function?
     c='r':m+s×Rand ⍵                        ⍝ Random Variable
       ⍝   c=¯2:m s(16808⌶)'Normal'⍵             ⍝ Random Variable using Ron's random variable generator
     b←⍵<0.5                                ⍝ Prob < 0.5?
     z←1 2∘(snorm Inv)¨|⍵-b                 ⍝ Find Z from P
     m+s×z×¯1*b                             ⍝ Convert Z to X
 }

 poisson←{
          ⍝⍟ Copyright (c) 2024 by Stephen M. Mansour
          ⍝∇ Poisson Distribution
          ⍝∊ Distribution Function; Discrete
          ⍝⍺ Lambda S Mean Rate
          ⍝⍺ P  P  Probability of success
          ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
          ⍝⍵ N   N Number of items in a particular interval
          ⍝← P   P Probability of exactly X successes in that interval
          ⍝⍕ Syntax:  Lambda poisson N
          ⍝⍎ 0.104444863 ← 5  poisson 7
     x←⍵ ⋄ ⎕ML←3 ⋄ ⎕IO←0                  ⍝ Set system variables
     x≡'Type':'Discrete'                  ⍝ Get parameter
     m c←getParameters ⍺
     (⊂x)∊'Mean' 'Variance':m             ⍝ Obtain theoretical values
     x≡'Median':⌊m+1 0.02-.÷3 m           ⍝   "       "           "
     x≡'Mode':y⊃⍨</m ∇ y←(⌈¯1+m),⌊m       ⍝   "       "           "
     x≡'Skewness':m*-÷2                   ⍝   "       "           "
     x≡'Kurtosis':÷m                      ⍝   "       "           "
         ⍝str←'pdqr'[1 0 ¯1 ¯2⍳c],'pois(⍵,⍵)'  ⍝ Build R Expression
     str←c,'pois(⍵,⍵)'                    ⍝ Build R Expression
     rAv:+##.∆r.x str ⍵ m                 ⍝ Call R if requested
     1<≢m:(m,¨c)∇¨⍵
     c='d':m{p←(*-⍺)×(⍺*⍵)÷!⍵⌈0           ⍝ Probability (X=x)
         p×(⍵=⌊⍵)∧⍵≥0}x                   ⍝ p=0 unless natural number
     c='p':(x≥0)×(⊂⌊0⌈x)⌷1{⍺>⌈/x:+\⍵      ⍝ Cumulative Pr(X≤x)
         (⍺+1)∇ ⍵,⍵[⍺-1]×m÷⍺},*-m         ⍝ Factor = Pr(x+1)÷Pr(x)
     c='q':+/x∘.>1{(+/⍵)≥⌈/x:+\⍵          ⍝ Inverse probability
         (⍺+1)∇ ⍵,⍵[⍺-1]×m÷⍺},*-m         ⍝ Factor = Pr(x+1)÷Pr(x)
     c='r':x⍴m{t←⍺'r'exponential⌈⍵×⍺+2×⍺*÷2 ⍝ Poisson Random Variable
         t0←+/(+\t)∘.>1+⍳⍵ ⋄ +⌿t0∘.=⍳⍵}×/x  ⍝
 }

∇ Z←ppchi2 Y;P;V;G;X;IFAULT;E;AA;XX;C;CH;A;P1;P2;T;Q;B;S1;S2;S3;S4;S5;S6;⎕ML;⎕IO
 ⎕ML←3 ⋄ ⎕IO←0
 IFAULT←0
 P V←Y
 G←⍟!¯1+0.5×V
 E←5E¯7
 AA←⍟2
 Z←¯1
 IFAULT←1
 →((P<0.000002)∨P>0.999998)/0
 IFAULT←2
 →(V≤0)/0
 IFAULT←0
 XX←0.5×V
 C←XX-1
 →(V≥¯1.24×*P)/L1
 CH←(P×XX×*G+XX×AA)*÷XX
 →(E<CH)↓L6 L4
L1:→(V>0.32)/L3
 CH←0.4
 A←⍟1-P
L2:Q←CH
 P1←1+CH×4.67+CH
 P2←CH×6.73+CH×6.66+CH
 T←¯0.5+(4.67+2×CH)÷P1
 T-←(6.73+CH×13.32+3×CH)÷P2
 CH-←((1-*A+G+0.5×CH+C×AA)×P2÷P1)÷T
 →(0.1<|Q÷CH-1)↓L4 L2
     ⍝ L3:X←GAUINV P IF1
L3:X←normal criticalValue≥P
 P1←0.222222÷V
 CH←V×(X×(P1*0.5)+1-P1)*3
 :If CH>6+2.2×V
     CH←¯2×(⍟1-P)+G-C×⍟0.5×CH
 :EndIf
L4:Q←CH
 P1←0.5×CH
     ⍝ P2←P-XX G gamma criticalValue≥P1
 P2←P-gamain P1 XX G IFAULT
 →(IFAULT=0)/L5
 IFAULT←3
L5:
 T←P2×*(XX×AA)+G+P1-⍟CH
 B←T÷CH
 A←(0.5×T)-B×C
 S1←(A⊥60 70 84 105 140 210)÷420
 S2←(A⊥1278 1141 966 735 420)÷2520
 S3←(A⊥932 707 462 210)÷2520
 S4←(A⊥1740 889 294 1182 672 252)÷5040
 S5←(84+(264×A)+C×175+606×A)÷2520
 S6←(120+C×346+127×C)÷5040
 CH←CH+T×1+0.5×T×S1-B×C×S1-B×S2-B×S3-B×S4-B×S5-B×S6
      ⍝CH←CH+T×1+0.5×T×S1-B×C×(-B)⊥S6 S5 S4 S3 S2 S1
 →(E<|1-Q÷CH)/L4
L6:Z←CH
∇

 rayleigh←{                       ⍝ Rayleigh Distribution
      ⍝⍟ Copyright (C) 2023 by Stephen M. Mansour
      ⍝∇ Rayleigh Distribution
      ⍝∊ Continuous Distribution Function
      ⍝⍺ S: Scale
      ⍝⍺ [0=Density, 1=Distribution, ¯1=Inverse Distribution, ¯2=random]
      ⍝⍵ X: Value
      ⍝← U: Density U≥0
      ⍝⍕ U ← S rayleigh X
      ⍝⍎ 0.0972 ← 5 2 rayleigh 1.3
      ⍝⍵ A partular value of the exponential random variable
     ⎕ML←3 ⋄ ⎕IO←0                       ⍝ Set system variables
     ⍵≡'Type':'Continuous'               ⍝
     pi←○1                     ⍝ Scale Parameters
     s c←getParameters ⍺
     ⍵≡'Mean':s×sqrt pi÷2                ⍝ Calculate theoretical values
     ⍵≡'Median':s×sqrt 2×⍟2              ⍝     "         "          "
     ⍵≡'Mode':s                          ⍝     "         "          "
     ⍵≡'Variance':(s*2)×0.5×4-pi
     ⍵≡'Skewness':2×(sqrt pi)×(pi-3)÷(4-pi)*3÷2
     ⍵≡'Kurtosis':-(pi⊥6 ¯24 16)÷(4-pi)*2
     c='d':0⌈(⍵÷s*2)×*-0.5×(⍵÷s)*2          ⍝ Density
     c='p':(⍵≥0)×1-*-0.5×(⍵÷s)*2            ⍝ Distribution
     c='q':s×sqrt-2×⍟1-⍵                   ⍝ Inverse
     1<≢s:(s,¨'r')∇¨⍵
     (s×sqrt 2)×(-⍟?⍵⍴0)÷2                ⍝ Random Variable
 }

 rectangular←{
       ⍝⍟ Copyright (c) by Stephen M. Mansour
       ⍝∇ Rectangular Distribution; Continuous Uniform Distribution
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ [A←0] X  Lower Bound
       ⍝⍺ [B←1] X  Upper Bound  (A≤B)
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [op] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ X   T   Probability, criticalValue or sampleSize
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← [A|_0 B|_1] rectangular [op [rel]] X
       ⍝⍎ 0.25←0 20 rectangular prob < 5               ⍝ Cumulative Prob
       ⍝⍎ 0 1 1 1 0 ←  rectangular ¯0.5 0 0.5 1 1.5    ⍝ U(0,1)
       ⍝⍎ 0 0.5 0.5 0.5 0 ← 2 4 rectangular 1 2 3 4 5  ⍝ U(2,4)
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ ⍺←0 1 0                ⍝ Default is U(0,1)
     ⍵≡'Type':'Continuous'
     ab c←getParameters ⍺
     a b←{2=≢⍵:⍵ ⋄ 0 ⍵}ab
         ⍝ a b←{⍵[⍋⍵]}a b                         ⍝ Sort lower to upper
     ⍵≡'Mean':(a+b)÷2                       ⍝ Calculate theoretical values
     ⍵≡'Variance':(2*⍨b-a)÷12               ⍝     "         "         "
     ⍵≡'Mode':⍳0                            ⍝     "         "         "
     ⍵≡'Skewness':0                         ⍝     "         "         "
     ⍵≡'Kurtosis':-6÷5                      ⍝     "         "         "
     str←c,'unif(⍵,⍵,⍵)' ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ a b                 ⍝ Use R if requested
     p←1≠×/×⍵∘.-a b                         ⍝ Is X∊[a,b] ?
     c='d':p÷b-a                              ⍝ Density
     c='p':0⌈1⌊(⍵-a)÷b-a                      ⍝ Cumulative Distribution
     c='q':a+⍵×b-a                           ⍝ Inverse
     1≠≡a b:(↑,/¨a b'r')∇¨⍵                ⍝ Multiple random vars
     c='r':a+(b-a)×?⍵⍴0                      ⍝ random Variable
     
     ⍝ c=¯3:                                     ⍝ GoodnessOfFit
     ⍝b←1≠×/×⍵-a
     ⍝     b÷d--/a                               ⍝ Density or probability
 }

 smirnov←{                                ⍝ Scalar,Arithmetic
          ⍝⍟ Copyright (c) 2022 by Stephen M. Mansour
          ⍝∇ Smirnov Distribution
          ⍝
          ⍝∊ Discrete Distribution Function
          ⍝⍺ N - Sample Size
          ⍝⍵ X - Value of statistic
          ⍝← Density?
          ⍝⍕ Syntax:  N kolmogorov X
          ⍝⍎ 0.975 ← 10 kolmogorov prob ≤ .409
     'Type'≡x←⍵:'Continuous'
         ⍝ M N C←3↑⍺ ⋄ ⎕ML←3 ⋄ ⎕IO←0                   ⍝ Set system variable
     N C←getParameters ⍺ ⋄ ⎕ML←3 ⋄ ⎕IO←0
         ⍝ N∨.<0 1:∆SigErr'All parameters must be positive'
     f←{=/⍺:÷/(N+N 0×⍵)!2×N←↑⍺
         ÷!/+\⍺}
     g←{=/⍺:#.NonParametric.A19{⍺[;0]⍳⍵}[;0]⍳↑⍺
     
         1-T×+/{P←T+⍵÷N ⋄ P÷⍨N P binomial prob=⍵}¨J}
     
         ⍝  x≡'Mean':M×N÷2                              ⍝ Theoretical Values
     ⍝     x≡'Variance':M×N×(M+N+1)÷12                 ⍝    "           "
       ⍝   ∨/~M N∊1+⍳10:∆SigErr'Domain Error'         ⍝ M,N integers between 1 and 10 inclusive
         ⍝ x≡'Mode':mode RF                            ⍝    "           "
     ⍝     x≡'Skewness':skewness RF                    ⍝    "           "
     ⍝     x≡'Kurtosis':kurtosis RF                    ⍝    "           "
     ⍝     str←'pdqr'[1 0 ¯1 ¯2⍳C],'wilcox(⍵,⍵,⍵)'     ⍝ Build R Expression
     ⍝    rAv:+##.∆r.x str ⍵ M N                      ⍝ Use R if requested
     1<≢⍵:⍺∘∇¨⍵                                  ⍝ Allow array arguments
     C='d':(-/N g¨⍵(+,-)h)÷2×h←0.00001             ⍝ Probability (X=x)
     C='p':÷/(N+N 0×⍵)!2×N                          ⍝ Cumulative Probability X≤x
     C='q':(⍵-N∘g)SecAlg D,⍵-N g¨D←1 0.7÷sqrt N   ⍝ Critical Values (Inverse)
     C='r':'Nonce Error'                          ⍝ Random Variables
 }

 swedEisenhart←{                                ⍝ Scalar,Arithmetic
          ⍝⍟ Copyright (c) 2021, 2024 by Stephen M. Mansour
          ⍝∇ Swed-Eisenhart Distribution
          ⍝  Annals of Mathematical Statistics 1943, Vol114. No. 1
          ⍝∊ Discrete Distribution Function
          ⍝⍺ M - number of items in Category A
          ⍝⍺ N - number of items in Category B
          ⍝⍵ R - Number of runs
          ⍝← Probability of X runs
          ⍝⍕ Syntax:  M N swedEisenhart R
          ⍝⍎ 0.0079 0.031746 ← 5 5 swedEisenhart 2 3
     'Type'≡x←⍵:'Discrete'
     ⎕ML←3 ⋄ ⎕IO←0
     (M N)C←getParameters ⍺ ⍝ Set system variables
     ∨/∊M N<0 1:∆SigErr'All parameters must be positive'
     f←{K←⌈⍵÷2 ⋄ ~2|⍵:×/2,(K-1)!⍺-1              ⍝ Even Combinations
         +/×⌿0 1⌽(K-1 2)∘.!⍺-1}                  ⍝ Odd Combinations
     x≡'Mean':1+2×(M×N)÷M+N                      ⍝ Theoretical Values
     x≡'Variance':(M+N){⍵×(⍵-⍺)÷(⍺*2)×⍺-1}2×M×N  ⍝    "           "
     x≡'Mode':{⍵⍳⌈/⍵}M N∘f¨⍳M+N+1                ⍝    "           "
     x≡'Skewness':'Nonce Error'                  ⍝    "           "
     x≡'Kurtosis':'Nonce Error'                  ⍝    "           "
     ⍝    str←'pdqr'[1 0 ¯1 ¯2⍳c],'hyper(⍵,⍵,⍵,⍵)'    ⍝ Build R Expression
     ⍝    rAv:+##.∆r.x str ⍵ m(N-m)n                  ⍝ Use R if requested
     ⍝     1<≢⍵:⍺∘∇¨⍵
     (1≠≡M N ⍵)∧C≠'r':(↑,¨/M N C)∇¨⍵                                  ⍝ Allow array arguments
     C='d':(M N f ⍵)÷M!M+N                         ⍝ Probability (X=x)
     C='p':(M!M+N)÷⍨+/M N∘f¨2↓⍳⍵+1                 ⍝ Cumulative Probability X≤x
     C='q':(⍵≥0.5)+⍵⍸⍨(M!M+N)÷⍨+\M N∘f¨⍳M+N+2     ⍝ Critical Values (Inverse)
     C='r':{↑runs ⍵[?⍨≢⍵]}¨⍵⍴⊂M N/0 1             ⍝ Random Variables
 }

 tDist←{                ⍝ t Distribution
      ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ Student t Distribution
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ DF     N  Degrees of Freedom
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍥ [operator] oD
       ⍝⍵⍵ [rel]  fR  Only for prob or critVal operators
       ⍝⍵ Value  S Test Statistic, probability or Sample Size
       ⍝← Result   S  Density, Probability or Critical Value
       ⍝⍕ Result ← DF tDist [operator [rel]] Value
       ⍝⍎ 2.015← 5 tDist critVal < .05 ⍝  T-table Value, one-sided
       ⍝⍎ ¯2.5706 2.5706←5 tDist critVal between .05 ⍝ T-Table value Two-sided
       ⍝⍎ 0.34886←5 tDist prob > 2.3            ⍝ P-Value; one-sided
       ⍝⍎ 0.069772←5 tDist prob outside 2.3     ⍝ P-Value; two-sided
       ⍝⍎ 0.043551←5 tDist 2.3                  ⍝ Density
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ w←⍵                           ⍝ Set system variables
     w≡'Type':'Continuous'
          ⍝m c←2↑⍺
         ⍝ m←{↑⍣(1=≢⍵)⊢⍵}(-1≥c←↑⌽⍺)↓⍺
     m c←getParameters ⍺
        ⍝  c←c×c≤1                             ⍝ Parameter (D.F)
        ⍝ c←checkOperator
         ⍝ w≡'Mean':{⍵>1:0 ⋄ ⎕NULL}m          ⍝ Calculate theoreetical value
     w≡'Mean':(⎕NULL 0)[m>1]             ⍝ Calculate
         ⍝ w≡'Variance':{⍵>2:⍵÷⍵-2 ⋄ ⎕NULL}m  ⍝    "          "          "
     w≡'Variance':⎕NULL@{m≤2}m÷1⌈m-2     ⍝    "          "          "
     w≡'Mode':0                          ⍝    "          "          "
     w≡'Skewness':(⎕NULL 0)[m>3]
         ⍝w≡'Kurtosis':{⍵≤4:⌊/⍳0 ⋄ 6÷⍵-4}m    ⍝    "          "          "
     w≡'Kurtosis':{⌊/⍳0}@{⍵≤4}6÷¯4+m+m=4 ⍝    "          "          "
         ⍝ 1≠≢m:∊⍣(c≠¯2)⊢(m,¨c)∇¨⍵             ⍝ Make pervasive
     1≠≢m:∊⍣(c≠'r')⊢(m,¨c)∇¨⍵             ⍝ Make pervasive
     str←c,'t(⍵,⍵)'                      ⍝ Build R expression
     rAv:+##.∆r.x str ⍵ m                ⍝ Use R if requestedion
     TDEN←{Z←÷⌿!¯1+0.5×1 0∘.+K←⍺⌊342     ⍝ Density Function
         Z←Z÷(○K)*÷2
         Z÷(1+(⍵*2)÷K)*(K+1)÷2}
     INTEGRAL←{AB←¯2↑0,⍵                 ⍝ Definite Integral
         H←(-/⌽AB)÷N←⍺
         XI0←+/⍺⍺ AB
         XI←((N÷2),2)⍴0,⍺⍺ H×1↓⍳N
         H×(1 2 4+.×XI0,+⌿XI)÷3}
     PRIME←{∆X←0.000001×⌊/X←,⍵                      ⍝ Derivative
         Z←(2⍴⍴X)⍴X
         Z←⍺⍺ X,[0]Z+∆X×N∘.=N←⍳⍴X
         ⍉((1 0↓Z)-Z[(⍴X)/0;])÷∆X}
     INV←{TOL←0.0001 ⋄ ⍺←1 ⋄ ⎕DIV←1                 ⍝ Inverse Operator
         FX←(⍺⍺ ⍺)-⍵
         X←⍺-FX÷⍺⍺ PRIME ⍺
         TOL<⌈/|X-⍺:X ∇ ⍵ ⋄ X}
     SNORM←{d←0.3193815 ¯0.3565638 1.781478 ¯1.821256 1.330274
         |(⍵≥0)-(0.3989423×*¯0.5×⍵*2)×d+.××⍀(5,⍴⍵)⍴÷1+0.2316419×100⌊|⍵}
     TDIST←{⍺>1001:SNORM ⍵                          ⍝ Cumulative Distribution
         0.5+(×⍵)×100(⍺∘TDEN)INTEGRAL¨|⍵}
     TINV←{↑¨(⍺∘TDIST)INV¨⍵}                        ⍝ Critical Value
     c='d':m{⍺=1:÷○(1+⍵)*2 ⋄ ⍺ TDEN ⍵}w               ⍝ Density
     c='p':m{##.Koko.Distrib.Student_P ⍵,⌊⍺}¨w
     c='q':m{⍵<0.5:-##.Koko.Distrib.Student_Tad ⍵,⌊⍺
         ##.Koko.Distrib.Student_Tpd ⍵,⌊⍺}¨w
     Y←normal randomVariable w                      ⍝ Normal Random Variable
     Z←m chiSquare randomVariable w                 ⍝ Chi Square random Variable
     Y÷(Z÷m)*÷2                                     ⍝ t Random Variable
 }

 triangular←{
      ⍝⍟ Copyright (c) 2024 by Stephen M. Mansour
      ⍝∇ Triangular Distribution; subjective description of a population with limited smaple data
      ⍝∊ Distribution; Discrete
      ⍝⍺ Lower    X Lower Bound
      ⍝⍺ Middle   X Mode     (Lower≤Middle≤Upper)
      ⍝⍺ Upper    X Upper Bound
      ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
      ⍝⍵ Value    X  Critical Value, Probability or Sample Size
      ⍝← Result   X  Density, Probability or Critical Value
      ⍝⍕ Result← Lower Middle Upper triangular [op [rel]] Value
      ⍝⍎ 0.16667 ← 2 4 5 triangular prob < 3         ⍝ Cumulative Probability
      ⍝⍎ 0.33333 ← 2 4 5 triangular 3                ⍝ Density
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ ⎕DIV←1                     ⍝ Set system variables
     ⍵≡'Type':'Continuous'                      ⍝ Continuous Distribution
        ⍝  a m b c←4↑⍺                                ⍝ Parameters
     (a m b)c←getParameters ⍺
        ⍝  a m b←{⍵[⍋⍵]}a m b                         ⍝ Sort them
     ⍵≡'Mean':(a+b+m)÷3                         ⍝ Calculate theoretical value
     ⍵≡'Variance':{18÷⍨(⍵+.*2)-⍵+.×1⌽⍵}a b m    ⍝     "          "        "
     ⍵≡'Mode':m                                 ⍝     "          "        "
     ⍵≡'Kurtosis':-0.6                          ⍝     "          "        "
     ⍵≡'Skewness':{z←×/(2×⍵)-(1⌽⍵)+2⌽⍵          ⍝     "          "        "
         z÷←((⍵+.*2)-⍵+.×1⌽⍵)*1.5
         z×(2*0.5)÷5}a m b
     c='d':0⌈{⍵≤m:2×(⍵-a)÷(b-a)×m-a               ⍝ Density
         2×(b-⍵)÷(b-a)×b-m}¨⍵                   ⍝
     1≠≡a m b:(↑,¨/a m b)∇¨⍵
     c='p':{⍵≤m:(2*⍨⍵-a)÷(b-a)×m-a                ⍝ Distribution
         (m b÷.-a)+(-/2*⍨b-m ⍵)÷b×.-a m}¨a⌈b⌊⍵  ⍝
     c='q':{⍵≤(m-a)÷b-a:a+0.5*⍨⍵×(m-a)×b-a       ⍝ Inverse
         b-0.5*⍨(1-⍵)×(b-a)×b-m}¨⍵              ⍝
     c='r':⍵⍴{u←?d←(×/⍵)⍴0                       ⍝ random Variable
         j←~i←u<(m-a)÷b-a                       ⍝
         (i/d)←a+0.5*⍨(i/u)×(m-a)×b-a           ⍝
         (j/d)←b-0.5*⍨(1-j/u)×(b-a)×b-m ⋄ d}⍵   ⍝
     c=¯3:'Not yet implemented'                 ⍝ GoodnessOfFit
 }

 trigonometric←{                                   ⍝ Average
     ⎕ML←3 ⋄ ⎕IO←0
     ⍵≡'Type':'Continuous'  ⍝ Continuous Distribution
     ⍺←0
     c←↑⌽⍺                   ⍝ Parameters
     a c←getParameters 0
     ⍵≡'Mean':0             ⍝ Mean attribute
     ⍵≡'Variance':○÷3       ⍝ Variance attribute
     f←{(1+2○⍵)÷○2}
     F←{0.5+(⍵+1○⍵)÷○2}
     I←{⍵{⍵-⍺(f÷F-⊣)⍵}⍣=0.5+0.25××⍵-0.5}
     
     c='d':f ⍵                ⍝ Density
     c='p':F ⍵
     c='q':I ⍵               ⍝
     c='r':I?⍵⍴0            ⍝ random Variable
     c='g':'Not yet imple⍵mented'                  ⍝ GoodnessOfFit
 }

 uniform←{                        ⍝ Average
          ⍝⍟ Copyright (c) 2024 by Stephen M. Mansour
          ⍝∇ Discrete Uniform Distribution; for continuous uniform, use rectangular
          ⍝∊ Distribution; Discrete
          ⍝⍺ [Lower←1] J Lower Bound
          ⍝⍺ Upper     J Upper Bound; must be greater than Lower
          ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
          ⍝⍵ Value     J  Value
          ⍝← Q         P  Probability: 1/(B-A-1)
          ⍝⍕ Q←[Lower|_1] Upper uniform Value
          ⍝⍎ 0.2 ← 5 uniform 2      ⍝ Lower bound defaults to 1
          ⍝⍎ 0.1 ← 11 20 uniform 17
     ⎕ML←3 ⋄ ⎕IO←0                    ⍝ Set system variables
     ⍵≡'Type':'Discrete'
     ab c←getParameters ⍺
     a b←{2=≢⍵:⍵ ⋄ 1 ⍵}ab
     n←1+b-a                          ⍝ Number of possible values
     ⍵≡'Mean':(a+b)÷2                 ⍝ Calculate theoretical values
     ⍵≡'Variance':(¯1+n*2)÷12         ⍝       "        "         "
     ⍵≡'Mode':a+⍳1+b-a                ⍝       "        "         "
     ⍵≡'Skewness':0                   ⍝       "        "         "
     ⍵≡'Kurtosis':-6 5÷.×1 ¯1+n*2     ⍝       "        "         "
     c='d':({(⍵≤0)∧⍵=⌊⍵}×/⍵∘.-a,b)÷n  ⍝ Probability
     c='p':0⌈1⌊(-a-1+⌊⍵)÷n            ⍝ Cumuluative probability
     c='q':a+⌈n×⍵                     ⍝ Inverse
     1≠≡a b:(↑,¨/a b c)∇¨⍵
     c='r':a+?⍵⍴n                     ⍝ randomVariable
 }

 weibull←{                       ⍝ Weibull Distribution
      ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
       ⍝∇ Weibull Distribution; Reliability Model
       ⍝∊ Distribution Function; Continuous
       ⍝⍺ Shape  S  Controls the shape of the distribution
       ⍝⍺ Scale  T  Characteristic Life; 63.2 percentile
       ⍝⍺_ C Cs ['d'=Probability, 'p'=Cumulative Prob, 'q'=Inverse Distribution, 'r'=random]
       ⍝⍵ Time      T  Lifetime or Age
       ⍝⍥ Op        oD
       ⍝⍵⍵ Rel      fR
       ⍝← Result    S  Density, Probability or Critical Value
       ⍝⍕ Result ← Shape Scale weibull [op [rel]] Time
       ⍝⍎ 1.7977E308 0.0034865 0.0018394 0.0011996 0.00085955 0.00065061←0.5 100 weibull 0 50 100 150 200 250 ⍝ Shape < 1; PDF decreases from infinity
       ⍝⍎ 1 0.0060653 0.0036788 0.0022313 0.0013534 0.00082085←1 100 weibull 0 50 100 150 200 250  ⍝ Shape = 1; PDF decreases from 1/characteristic life
       ⍝⍎ 0 0.0046971 0.014715 0.00085451 3.6011E¯8 6.7803E¯18←4 100 weibull 1 50 100 150 200 250  ⍝ Shape > 1; PDF increases from 0,then decreases
       ⍝⍎ 0 0.060587 0.63212 0.99367 1←4 100 weibull prob ≤ 0 50 100 150 200 ⍝ CDF; F(t)
       ⍝⍎ 1 0.93941 0.36788 0.0063297 1.1254E¯7←4 100 weibull prob > 0 50 100 150 200 ⍝ Reliability or Survivor Function R(t)=1-F(t)
     ⎕ML←3 ⋄ ⎕IO←0                          ⍝ Set system variables
     ⍵≡'Type':'Continuous'                  ⍝
         ⍝m k c←3↑⍺                              ⍝ Scale, Shape Parameters
     (m k)c←getParameters ⍺
     ⍵≡'Mean':k×!÷m                         ⍝ Calculate theoretical values
     ⍵≡'Median':k×(⍟2)*÷m                   ⍝     "         "          "
     ⍵≡'Mode':(m≥1)×k×(1-÷m)*÷m             ⍝     "         "          "
     ⍵≡'Variance':(k*2)×(!2 1÷m)-.*1 2
     ⍵≡'Skewness':m{x←⍺ ⍵ weibull'Mean'
         v←⍺ ⍵ weibull'Variance'
         (v*-3÷2)×((!3÷⍺)×⍵*3)-(3×x×v)+x*3}k
     ⍵≡'Kurtosis':'Nonce Error'
     2=≡m k:(↑,¨/⍺)∇¨⍵                        ⍝ PervasiveLeft argument
     str←c,'weibull(⍵,⍵,⍵)'
          ⍝#.R_Available:+#.∆r.x str ⍵ m k
     rAv:+##.∆r.x str ⍵ m k
          ⍝c=0:(m÷⍵)×((⍵÷k)*m)×*-(⍵÷k)*m         ⍝ Density
     c='d':{z←((t÷k)*m)×(*-(t÷k)*m)×m÷t←⍵⌈⎕CT ⍝ Density
         z←(⌊/(1+×m-1)↑1 0)@(⍸⍵=0),z        ⍝ Replace f(0) with ¯ 1 or 0
         (⍴⍵)⍴z×⍵≥0}⍵                       ⍝ P(X<0)=0
     c='p':(⍵≥0)×1-*-(⍵÷k)*m                  ⍝ Distribution
     c='q':k×(-⍟1-⍵)*÷m                      ⍝ Inverse
     k×(-⍟?⍵⍴0)÷m                           ⍝ Random Variable
 }

 wilcoxonT←{                                ⍝ Scalar,Arithmetic
          ⍝⍟ Copyright (c) 2021, 2025 by Stephen M. Mansour
          ⍝∊ Table A12 QUantiles of Wilcoxon Signed Ranks Test
          ⍝∊ Conover, Practical Nonparametric Statistics Table 12, p. 545
          ⍝∇ Wilcoxon T distribution
          ⍝∊ Distribution Function; Discrete
          ⍝⍺ N N Sample Size
          ⍝⍵ R N Value of statistic
          ⍝← P P Probability
          ⍝⍕ P ← N wilcoxonT R
          ⍝⍎ 0.0079 0.031746 ←  5 wilcoxonT 2 3
     'Type'≡x←⍵:'Discrete'
     ⎕ML←3 ⋄ ⎕IO←0                                  ⍝ Set system variables
     N C←getParameters ⍺
     1≠≢N:(N,¨C)∇¨⍵                                 ⍝ Multiple
     M←0.25×N×N+1                                   ⍝ Mean of distribution
     RF V←{⍵>10:0,24÷⍨⍵×(⍵+1)×1+2×⍵                 ⍝ Large sample variance
         RF←relative 1 frequency(1+⍳⍵)+.×(⍵/2)⊤⍳2*⍵ ⍝ Small sample frequency
         RF(var RF)}N                               ⍝   and variance
     x≡'Mean':M                                     ⍝ Theoretical Values
     x≡'Variance':V                                 ⍝    "           "
     x≡'Mode':M                                     ⍝    "           "
     x≡'Skewness':0                                 ⍝    "           "
     x≡'Kurtosis':N{⍺>10:0 ⋄ kurtosis ⍵×⍤1⊢1,2*⍺}RF ⍝    "           "
     S←sqrt V                                       ⍝ Standard deviation
     C='d':N{⍺>10:M S normal ⍵ ⋄ RF[⍵;1]}⍵          ⍝ Density
     C='p':N{⍺>10:M S normal prob≤⍵+0.5             ⍝ Cumulative Prob
         +/(⍵+1)↑RF[;1]}⍵                           ⍝   Small sample
     C='q':N{⍵=0.5:M
         ⍺≤10:1+(+\RF[;1])⍸⍵
         ⍵<0.5:⌊M S normal critVal>⍵                ⍝ Critical Value
         ⍵>0.5:⌈M S normal critVal>⍵}¨⍵             ⍝   Small sample
     C='r':(?(⍵,N)⍴2)+.×1+⍳N                        ⍝ Random Variables
     
     
 }

 trunc←{p←¯1↓⍺ ⋄ op←↑⌽⍺
     x←⍵{{1⊃⍵[⍋⍵]}⍺,⍵}⍵⍵
     op=0:(p ⍺⍺ x)÷p ⍺⍺ prob between ⍵⍵
     op=1:↑÷/⍺⍺{p ⍺⍺ between ⍵}¨(⍵⍵⌊x)⍵⍵
     op=¯1:2+
     op=2:
 }


:EndNamespace 
