﻿:Namespace Regression
⍝ === VARIABLES ===

int←1


⍝ === End of variables definition ===

(⎕IO ⎕ML ⎕WX)←1 1 3

∇ Z←MakeFn X
 Z←X∘{⍺+2×⍵}
 Z
∇

 boxCox←{⍝BoxCox Transformation
      ⍝∇ Box Cox Model
      ⍝∊ Section 3.7.1 p. 94
      ⍝∊ Montgomery Into to Regression Analysis
      ⍝⍺ FV Response Variable
      ⍝⍵ FV Predictor Variable
      ⍝← FI Lambda
     f←{YDot←*(+/⍟⍵)÷≢⍵
         ⍺=0:YDot×⍟⍵
         (¯1+⍵*⍺)÷⍺×YDot*⍺-1}
     L←¯2 ¯1 ¯0.5 0 0.5 1 2
     YL←L f¨⊂⍺
     M←YL{⍺ regress ⍵}¨⊂⍵
     SSE←M.(DF×MSE)
     L[SSE⍳⌊/SSE]}

 buildRegressionEquation←{
     ⍝⍺ Name List
     ⍝⍵ Coefficients
     ⍝← Format
     ⎕IO←0 ⋄ ⎕ML←3 ⋄ ⎕PP←5             ⍝ set system variables
     w←,∘⎕FMT¨|⍵
     s←'+-'[⍵<0]
     statEq←{(≢⍺)>≢⍵:(↑⍺)'=',,s,⍵,⍪1↓⍺        ⍝ No intercept?
         z←(('+'=↑s)↓↑s),(↑⍺)'='(↑⍵)          ⍝ indep var, intercept
         z,,1↓s,⍵,⍪⍺}                         ⍝ slopes
     APLEq←{b←,∘⎕FMT¨⍵
         i←>/≢¨⍺ ⍵                            ⍝ Intercept excluded?
         z←(↑⍺),'←',(~i)/↑b                   ⍝ Start expression
         a←{')('≡2↑x←¯1⌽⍵:2↓x ⋄ ⍵}¨⍺          ⍝ Remove redundant parens
         z←∊z,i↓,'+','(',1⌽')',(~i)↓b,'×',⍪i↓a
         z,'+E'} ⍝ Build Expression
     ⍺ APLEq ⍵                                ⍝ Default is APL expression
     z←⍺ statEq w                             ⍝ Math equation
     z←deleteExcessBlanks,⍕z                  ⍝ Flatten
     ⊃((-⌈0.5×z⍳'=')↑'∧')z                    ⍝ Add hat for estimate
 }

 correl←{⎕IO←0 ⋄ ⎕ML←3
     x←⊃⍵
     n←↑⌽⍴x
     sx←+/x
     ss←(n×x+.*2)-sx*2
     sxy←+.×pairwise ⍵
     sxsy←×pairwise sx
     z←(n×sxy)-sxsy
     z÷(×pairwise ss)*÷2
 }

 cov←{
      ⍝⍝⍟ Copyright 2018 by Stephen M. Mansour
      ⍝∇ Covariance
      ⍝∊ Summary Function
      ⍝⍺ Y simple vector
      ⍝⍵ X simple or nested vector
      ⍝← Sample covariance
      ⍝← Numeric Scalar if X and Y simple numeric vectors
      ⍝← Numeric Matrix if monadic
      ⍝← Matrix with labels if left argument is a namespace
      ⍝←  and right argument is a character vector
      ⍝⍕ Z←Y cov X
      ⍝⍕ Z←cov X1 X2 ... Xn
      ⍝⍕ Z←cov N 'X1 X2 ... Xn'
      ⍝⍎ Height cov Weight
      ⍝⍎ cov Height Weight ShoeSize
     ⎕IO←0 ⋄ ⎕ML←3
     0=⎕NC'⍺':∘.∇⍨⍵              ⍝ No left arg, make cov matrix
     9=⎕NC'⍺':⍺ ∇{m←⍺⍺ ⍺.(⍎⍵)    ⍝ If namespace
         m{(⍵,⍨⊂'cov'),⍵⍪⍺}(⍵≠' ')⊂⍵}⍵   ⍝ Add labels
     0J1∊⍺:(1↓⍺)showCov ⍵        ⍝ Display covariance
     1∊≢¨⍺ ⍵:0                   ⍝ 0 if scalar
     rAv:+##.∆r.x'cov(⍵,⍵)'⍺ ⍵   ⍝ R calculation
     n←≢⍵                        ⍝ Sample Size
     z←(n×⍺+.×⍵)-×/+/⊃⍺ ⍵        ⍝ Sum of Squares
     z÷n×n-1                     ⍝ Covariance
 }

 covariance←{
     0=⎕NC'⍺':cov ⍵
     ⍺ cov ⍵
         ⍝ ⎕ML←3 ⋄ ⎕IO←0
     ⍝     n←1⊃⍴XX←⊃⍉⍵
     ⍝     SXSY←∘.×⍨+/XX
     ⍝     SXY←+/¨∘.×⍨↓XX
     ⍝     ((n×SXY)-SXSY)÷n×n-1
     
 }

 cubic←{⍝ Generate cubic centralmodel
     ⍵≡4:5
     ⍵≡1:3
     (⍵-mean ⍵)∘.*1 2 3
          ⍝⍝⍝⍝⍝⍝⍝ New Version ⍝⍝⍝⍝⍝⍝
     Int←(1=⊃⍵)∧2=≡⍵
     ⍺ 3(Int regressPoly)Int↓⍵
     
 }

 fitted←{
      ⍝ ∇Estimate fitted values
      ⍝ ∊ Regression Analysis, Montgomery & Scott pp. 109-118
      ⍝ ⍺   Parameter List (Optional)
      ⍝ ⍵ = Dependent and Independent Variables (y), x[1], x[2] ... x[n]
      ⍝ ← = Coefficient Estimates (fitted values yhat[1],...yhat[n])
     ⎕ML←3 ⋄ ⎕IO←0
     ⍺←⍬ ⋄ d←⍺≡⍬                    ⍝ Parameters on left?
     b←⍺{d:regress ⍵ ⋄ ⍺}⍵          ⍝  Estimate parameters
     b+.×X←1⍪d↓{,[⍳⍴⍴⍵],[¯0.5]⍵}⊃⍉⍵ ⍝ Predictor Variables
 }

 getWeights←{
     ⍝∇   Get weights based on inverse variance
     ⍝∊  Mongtomery & Peck 3.8 Weighted Least Squares p. 100
     ⍝⍺:  Grouping Parameter (See Frequency)
     ⍝⍵:  Response, Predictor Variables
     ⍝←:  Weights
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ ⍺←5
     Y X←↓⍉⍵
     FR←⍺ frequency X
     BP←(⌊/,⌈/)⍬
     BP←1⌽BP,0.5×2+/FR[;0]              ⍝ Breakpoints
     B←X between BP                   ⍝
     XBAR←(X+.×B)÷+⌿B
     S2←var¨(↓⍉B)/¨⊂Y
     XBAR S2←(~S2∊⎕NULL)∘/¨XBAR S2
     Beta←S2⌹1,⍪XBAR                  ⍝ Relation bet XBar and S2
     ÷(⍪X)⊥⌽Beta}

 leastSquares←{
      ⍝⍟ Copyright (C) 2018, 2023 by Stephen M. Mansour
      ⍝∇ Linear Regression using Least Squares method
      ⍝∊ Regression
      ⍝⍺ B: Include Intercept? or Weights
      ⍝⍵ Y Dependent and Independent Variables y, x[1], x[2] ... x[n]
      ⍝⍵ X1: Independent Variable 1
      ⍝⍵ Xn: Independent variable n
      ⍝← NS:   Namespace containing the following variables
      ⍝ ← = Coefficient Estimates (float) b[0], b[1], ... b[n]
      ⍝ ← = Standard Errors (float)
      ⍝ ← = t-Values (float)
      ⍝ ← = p-Values (float)
      ⍝     AnovaTable  RSq  RSqAdj  S  Type
      ⍝⍕ NS ← LeastSquares Y X1 [X2 ... [Xn]]
      ⍝⍎ NS ← LeastSquares Height ShoeSize                      ⍝ Simple Regressionsion
      ⍝⍎ NS ← LeastSquares Height ShoeSize (Sex eq 'M')         ⍝ Multiple Regression
      ⍝⍎ NS ← LeastSquares D 'Height ShoeSize (Sex eq ''M'')'   ⍝  "          " with Names
      ⍝⍎ NS ← LeastSquares M
      ⍝⍎ NS ← LeastSquares NS
     ⎕ML←3 ⋄ ⎕IO←0 ⋄ ⎕DIV←1 ⋄ ⍺←1 ⋄ int←×↑(⍺,1)~0J1   ⍝ Set system variables, defaults
     9=⎕NC'⍵':∇{EX←⍕,{(⍵[0]∊⎕A)∧∧/⍵∊∊iotag¨'Zz9':⍵    ⍝ If simple variable name, done
             '(',⍵,')'}¨⍵.(Select/Expressions)        ⍝ Else surround with parens
         ⍵.Intercept ⍺⍺ ⍵.Database EX}⍵               ⍝ If simple namespace, assemble expression
     NM W NL←{y←↑⍵                                    ⍝ Get data and name list
         9=⎕NC'y':1,y{⍺.(⍎⍵)⍵}↑⌽⍵                     ⍝ If NS, get variable names
         2=⍴⍴⍵:0,(↓⍉⊃⍵)(1↓,'< B>,G<9>'⎕FMT⍳↑⌽⍴⍵)      ⍝ IF matrix, columns are variables
         0,⍵(1↓,'< B>,G<9>'⎕FMT⍳⍴⍵)}⍵                 ⍝ If data, use B0, B1 ... Bn
         ⍝ W←(~∨/W∊¨⎕NULL)/¨W                               ⍝ Remove ⎕nulls
     W←removeNulls W
     rAv:⍺ regressR W
     ns←⍺{(⍺≢1)∧⍺≢0:⍺ regressW ⍵
         int runRegress ⍵}W
     NL←{')('≡2↑¯1⌽⍵:1↓¯1↓⍵ ⋄ ⍵}¨((' '≠NL)∨≠\NL∊'()')⊂NL ⍝ Parse Name List
     CC←(int/⊂'Intercept'),1↓NL                       ⍝ Coefficient List
     d←(⊃CC),' ',('2F12.4,2F10.5'⎕FMT ns.(B,SB,⍪T)),'  '∘fmtP⍤0⊢ns.P ⍝ coefficients
     H←,¨'Variable' 'Coeff' 'SE' 'T' 'P'              ⍝ Append Headers
     d←d⍪⍨∊((1+⌈/≢¨CC),2/-12 10)↑¨H                   ⍝    to data
     ns.Coeff←H⍪CC,ns.(B,SB,T,⍪P)                     ⍝ Coefficient table
     ns.Type←'Regress'                                ⍝ Object type is "Regress"
     ns.f←(ns.(B DF S C),int)linear
     NL←NM{⍺:⍵ ⋄ 'Y',('B'⎕R'X')¨1↓⍵}NL                ⍝ Change Bn to Xn
     ns.Equation←NL buildRegressionEquation ns.B      ⍝ yhat = b0 + B1 x1 + ...
     ns.VariableExpressions←NL                        ⍝ List of variable names
     ns.VariableNames←∪↑,/getNames¨NL                 ⍝ Generate Report ↓↓↓
     ns._Report←ns.AnovaTable{4=≢⍺:⍵                  ⍝ If Lack of Fit/Pure Error
         D←(1↓⍺[0;])formatAnova 1 1↓⍺                 ⍝ Reformat anova table
         H←⍕⊃(~B←D[;1]='-')\⍺[;0]                     ⍝ Reformat Source Names
         (B⌿H)←'-' ⋄ ⍕⍪(H,D)(¯2↑⍵)}ns._Report        ⍝ Reassemble Formatted Anova Table
     ns._Report←⍕⍪'The regression equation is:' ' 'ns.Equation' 'ns._Report' ' 'Solution' ' 'd
     ns.VariableTypes←NM{(d v)←⍵
         ⍺:{1<≢⍵:0≢↑0⍴⍵}¨⍎(1<≢v)↓'⊂',⍕'d.('v')'       ⍝ Get variable types
         (≢v)⍴0}(↑⍵)ns.VariableNames                  ⍝ 1=char, 0=numeric
     ns.Graph←regressionPlot ns                       ⍝
     ns.⎕IO←0
     _←ns.⎕DF ns.Equation                 ⍝
     (~0J1∊⍺):ns                                      ⍝ Generate namespace
     showRegress ns                                   ⍝
 }

 logistic←{
     ⍝∊ R Febrianti, et al 2021 J. Phys: Conf Series 1725 012014
     ⍝∇ Calculate logistic regression model
     ⍝⍺ Y - Response Variable (Boolean)
     ⍝⍺ X - Predictor Variable
     ⍝⍵ Intercept (Guess)
     ⍝⍵ Slope  (Guess)
     ⍝← Coefficients
     ⍝← Function
     ⍝⍎ NS←(Y,⍪X) logistic Intercept Slope
     
     ⎕IO←0 ⋄ ⎕ML←1             ⍝ Initialize
     Y X←↓⍉⍺                   ⍝ Response, predictor variables
         ⍝ Y←⍺[;0]  ⋄ X←0 1↓⍺
     Beta←⊃⊆⍵
     Pi←÷1+*-(⍪X)⊥⌽Beta
          ⍝Pi←÷1+*-X+.×Beta
        ⍝  U←(+/Y-Pi)(+/X×Y-Pi)
     A←+/Pi×1-Pi
     B←+/X×Pi×1-Pi
     C←+/(X*2)×Pi×1-Pi
        ⍝  IB←-⌹2 2⍴A B B C
     JB←2 2⍴A B B C
        ⍝  ⎕←B←⍵-IB+.×U
        ⍝  B
     W←(∘.=⍨⍳≢Y)×⍤1⊢Pi×1-Pi
     XX←1,⍪X
     WX←(sqrt W)+.×XX
     JBINV←⌹JB
     H←0 0⍉WX+.×JBINV+.×⍉WX
     Xi←H×(Pi-0.5)÷Pi×1-Pi
     bB←JBINV+.×(⍉XX)+.×W+.×Xi
       ⍝  UB←U-JB+.×bB
     UU←(Y+H÷2)-Pi×1+H
     UB←(+/UU)(UU+.×X)
     Beta+←bB+JBINV+.×UB
     Beta JBINV H Pi
     
 }

 logistic1←{
     ⍝∇ Calculate logistic regression model
     ⍝⍺ Sample Sizes (Optional Vector)
     ⍝⍵ Response Variable (Boolean or Number of Successes)
     ⍝⍵ Predictor Variable
     ⍝← Coefficients
     ⍝← Function
     ⍝⍎ NS←N logistic C X
     ⍝⍎ NS←logistic B X
     ⎕IO←0 ⋄ ⎕ML←1 ⋄           ⍝ Initialize
     Y←⍺ ⋄ X←⍵
     p←{÷1+*-⍵⊥⌽⍺}
     f←{U←Y-÷1+*-(⍪X)⊥⌽⍵
         (U+.×1,⍪X)+.*2}
     f SecAlg 1 1,f 1 1
                ⍝ Vector
     NS←⎕NS''
     NS.B←V⌹M                  ⍝ Int, Slope Parameters
     NS.f←NS.B∘{÷1+*-⍵⊥⌽⍺}     ⍝ Function
     NS
 }

 logit←{
      ⍝ Logistic regression
     
     ⍺≠⍵}

 makeIndicator←{
     ⍝∇  Generate indicator variables from a character field
     ⍝⍺ Left argument indicates default item | 1 = Single Value
     ⍝⍵ Right argument is character vector
     ⍝← Indictor variable(s)
     ⎕IO←0 ⋄ ⎕ML←1
     D←{0=⊃⊃0⍴⍵:⍕⍪⍵ ⋄ toMatrix ⍵}⍵
     f←{~⍵∊' _',⎕D,(⊢,¯1∘⎕C)⎕A} ⍝ Replace illegal chars
     D←('_'@f)D            ⍝ with underscores
     D←~∘' '⍤1⊢D           ⍝ Remove Blanks
     D[;0]←1 ⎕C D[;0]      ⍝ First char upper case
     OK←D[;0]∊⎕A           ⍝ If 1st char numeric
     D←(-~OK)⌽D,'∆ '[OK]   ⍝ Append suitable char
     U←∪D                  ⍝ Unique character
     ⍺←U[0;]               ⍝ Base case is first unless specified
     B←↓⍉D∧.=⍉U            ⍝ Create booleans
        ⍝ B←~⍣(⍺≡¯1)⊢B          ⍝ If left arg = ¯1, flip it
     ⍺≡1:⊃B                ⍝ If left arg = 1, make boolean vector
     _←⍎⍕(↓U)'←B'          ⍝ Else assign to names in char field
         ⍝ _←⎕EX'#.∆NS'                 ⍝ Clear namespace
     _←'#.∆NS'⎕NS(↓U)~⊂(≢⍉U)↑⍺  ⍝ Put into namespace
     #.∆NS
 }

 makePolynomial←{
     ⍝∇  Generate polynomial variables from a character field
     ⍝⍺ Left argument indicates degree of Polynomial or X1
     ⍝⍵ Right argument is variable
     ⍝← Indictor variable(s)
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ ⍺←2          ⍝ Default is quadratic
     I←(1=≢⍺)∧(0∘<∧⌊=⊢)⊃⍺         ⍝ Is left argument integer?
     X←(⊢-+/÷≢)¨I↓⍺ ⍵             ⍝ Center the data
     K←I⊃2 ⍺                      ⍝ Degree of polynomial
     XX←(,⍉X∘.*1+⍳K),×pairwise X  ⍝ Generate variables
     NMS←,⌽⍉I↓'YX'∘.,K↑1↓⎕D       ⍝ Generate Names
     NMS,←I↓⊂'XY'                 ⍝
     _←⍎⍕NMS'←XX'                 ⍝ Assign values
     _←⎕EX'#.∆NS'                 ⍝ Clear namespace
     _←'#.∆NS'⎕NS NMS             ⍝ Create namespace
     #.∆NS
 }

 makeTransFn←{
     ⍝ Build a user transformation function
     ⍝ ⍵  2-Column matrix  [Names|Expressions] or vector of expressions
     ⍝ ←  Namespace containing function
     ⎕IO←0 ⋄ ⎕ML←3  ⍝⋄ NS←⎕NS''
        ⍝  NM XP←↓⍉⍵  ⍝ Names, Expressions
     NM XP←↓⍉{2=⍴⍴⍵:⍵ ⍝Names,Expressions
         ⊃(≠∘'←'⊂⊢)¨⍵}⍵
     CR←'transform←{' 'NS←⎕NS '''''
     CR,←⊂'NS.Int←1'
       ⍝   CR,←⊂'2=⍴⍴⍵:⊃⍺∘∇¨↓⍵'
     CR,←NM{ ⍝⍺≡'Int':'NS.Int←',⍵
         'NS.',⍺,'←⍵.(',⍵,')'}¨XP
     CR,←⊂'NS}'
     R←⎕FX CR
     MSG←'To run regression, enter the following:'
     ⍪MSG'report transform regress #.[Current Database]'
 }

 predict←{
     ⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
     ⍝∇ Predict fitted values/intervals for regression model
     ⍝∊ Regression
     ⍝⍺ Regression Model Namespace
     ⍝⍵  Array of inputs; each row represents a set of values
     ⍝←  n×5 matrix of outputs [Fitted Value, ConfInt, PredInt]
     ⍝   MODEL predict X
     ⎕IO←0 ⋄ ⎕ML←3 ⋄ M←⍺                    ⍝ Set system variables
     c←{2≠⍵.⎕NC'Confidence':0.95            ⍝ Conf = 95% if not specified
         0.01×⍵.Confidence}M                ⍝ Else get from namespace
     x←↓(⍉⍵){,¨@(⍵/⍳⍴⍵)⊢⍺}1↓M.VariableTypes ⍝ Assemble right argument
     ⍺.Confidence←100×c ⋄ ⍺.XValues←⍵       ⍝ Save conf level, X,YValues in NS
     0=⍴x:+⍺.YValues←1 5⍴0
     _←⍎⍕1↓M.VariableNames,'←x'             ⍝ Assign to variable names
     in←↓⍉⊃⍎¨1↓M.VariableExpressions        ⍝ Execute variable expressions
     y←⊃,/⊃c M.f{(⍺⍺ ⍵),(⍺ ⍺⍺ confInt ⍵),⍺ ⍺⍺ predInt ⍵}¨in
     y←y{1=↑⌽⍴⍵:⍉(5,≢⍵)⍴⍺ ⋄ ⍺}⍵             ⍝ If 1-col matrix, reshape
     ⊢⍺.YValues←y}

 quad←{quadratic ⍵}

 quadratic←{⍝ Generate quadratic central model
     ⎕IO←0 ⋄ ⎕ML←3
     ⍵≡4:5
     ⍵≡1:2
     (⍵-mean ⍵)∘.*1 2
          ⍝⍝⍝⍝⍝ New version ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝
     Int←(1=⊃⍵)∧2=≡⍵
     Y X←{(⊃⍵)(1↓⍵)}Int↓⍵
     Y 2(Int regressPoly)X
     
     
 }

 regress1←{
      ⍝ ∇Regression
      ⍝ ∊ Regression Analysis, Montgomery & Scott pp. 109-118
      ⍝ ⍺ Include Intercept?
      ⍝ ⍵ = Dependent and Independent Variables y, x[1], x[2] ... x[n]
      ⍝ ← = Coefficient Estimates (float) b[0], b[1], ... b[n]
      ⍝ ← = Standard Errors (float)
      ⍝ ← = t-Values (float)
      ⍝ ← = p-Values (float)
     ⎕ML←3 ⋄ ⎕IO←0      ⍝ Set system variables
     ⍺←1                ⍝ Include y-intercept?
     X←(~⍺)↓1⍪1↓⊃⍉⍵     ⍝ Predictor Variables
     n←≢y←↑⍵            ⍝ Sample size
     C←⌹X+.×⍉X          ⍝ Covariance Matrix
     C+.×X+.×y          ⍝ Estimate coefficients
 }

 regressPlot←{
      ⍝Create a Regression Plot
     ⎕ML←3 ⋄ ⎕IO←0
     A B←(⌊/,⌈/),⍺[;1]
     XF←A to B,(B-A)÷80       ⍝ Fitted X
     YF←XF{0::⍵.f ⍺ ⋄ ⍵.g ⍺}⍵ ⍝ Fitted Y
     YCI←XF{0::⍵.f confInt ⍺
         ⍵.g confInt ⍺}⍵
     ⍝    D←⍺⍪YF,⍪XF
     D←⍺⍪(,YF,YCI),⍪3/XF
     S←(⌽D(-,⊢)⍥≢⍺)/'*.'
         ⍝ Plot←S scatterPlot show↓⍉D
     scatterPlot show↓⍉D
     ⍕⍪Plot'* = Data          . = Fitted'
 }

 regressR←{
      ⍝⍟ Copyright (C) 2018, 2023 by Stephen M. Mansour
      ⍝∇ Linear Regression using R
      ⍝∊ Regression
      ⍝⍺ 1 = intercept, 0 = no intercept
      ⍝⍵ Y Dependent and Independent Variables y, x[1], x[2] ... x[n]
      ⍝⍵ X1: Independent Variable 1
      ⍝⍵ Xn: Independent variable n
      ⍝← NS:   Namespace containing the following variables
      ⍝ ← = Coefficient Estimates (float) b[0], b[1], ... b[n]
      ⍝ ← = Standard Errors (float)
      ⍝ ← = t-Values (float)
      ⍝ ← = p-Values (float)
      ⍝     AnovaTable  RSq  RSqAdj  S  Type
      ⍝⍕ NS ← regressR Y X1 [X2 ... [Xn]]
      ⍝⍎ NS ← regressR Height ShoeSize                ⍝ Simple regression
      ⍝⍎ NS ← regressR Height ShoeSize (Sex eq 'M')   ⍝ Multiple regression
     ⎕IO←0 ⋄ ⎕ML←3 ⋄ ns←⎕NS'' ⋄ ⍺←1              ⍝ Set system variables
     Y←↑⍵ ⋄ X←⍉⊃1↓⍵                              ⍝ Get dependent, indep variables
       ⍝   _←##.∆r.x'A <- lsfit(⍵,⍵, intercept = ⍵)'X Y(⍺⊃'FALSE' 'TRUE')⍝ Perform linear regression
     W←(⍺≢0)∧⍺≢1   ⍝ Weighted Least Squares?
     str←'A <-lsfit(⍵,⍵, ',(W⊃'intercept' 'wt'),'= ⍵)'
     _←##.∆r.x str X Y,⊂⍺{⍵:⍺ ⋄ ⍺⊃'FALSE' 'TRUE'}W
     INT←⍺≢0
         ⍝ str←,'G<,A$coefficients[[Z9]]>'⎕FMT 1↓⍳⍺+⍴⍵ ⍝ Build R expression
     str←,'G<,A$coefficients[[Z9]]>'⎕FMT 1↓⍳INT+⍴⍵ ⍝ Build R expression
     ns.B←##.∆r.x'c(',1↓str,')'                  ⍝ Get coefficients
     ns.E←##.∆r.x'A$residuals'                    ⍝ Get residuals
     _←##.∆r.x'D<-ls.print(A)'                   ⍝ Generate report from R
     CT←##.∆r.x'as.numeric(D$coef.table[[1]])'   ⍝ Get coefficient table
     CT←,¨'Coeff' 'SE' 'T' 'P'⍪⍉4(0.25×≢CT)⍴CT   ⍝ Include headers
         ⍝ ns.C←C←{⌹⍵+.×⍉⍵}(~⍺)↓1⍪1↓⊃⍉⍵                ⍝ Predictor Variables
     ns.C←C←{⌹⍵+.×⍉⍵}(~INT)↓1⍪1↓⊃⍉⍵                ⍝ Predictor Variables
         ⍝ ns.Coeff←((1 ⍺/'Variable' 'Intercept'),↓'P<B>LI2'⎕FMT ⍺↓(~⍺)+⍳≢ns.B),CT
     ns.Coeff←((1 INT/'Variable' 'Intercept'),↓'P<B>LI2'⎕FMT INT↓(~INT)+⍳≢ns.B),CT
     A←##.∆r.x'as.numeric(D$summary)'            ⍝ Obtain summary data from R
     ns.(S RSq)←2↑A                              ⍝ Get std err and R-Squared
     ns.DF←DF←-/≢¨ns.(E B)                             ⍝ Degrees of Freecom
     T←3 5⍴0                                     ⍝ Reconstruct Anova table
     T[0;3 4]←A[2 5]                             ⍝ F and P
     T[;1]←{⍵,+/⍵}A[3 4]                         ⍝ Degrees of Freedom
     T[1;2]←ns.S*2                               ⍝ Mean Squared Error MSE
     T[0;2]←T[1;2]×A[2]                          ⍝ Mean Squared Regression MSR
     T[;0]←{⍵,+/⍵}×/T[0 1;1 2]                   ⍝ Sum of Sauares
     T←'SS' 'DF' 'MS' 'F' 'P'⍪T                  ⍝ Headings
     ns.AnovaTable←'Source' 'Regression' 'Error' 'Total',T
     ns.(_Report←⍕,[⍬]AnovaTable ⍬ Coeff ⍬(2 2⍴'S' 'R-Sq'S('K2Q<%>F7.2'⎕FMT RSq)))
     ns.YHAT←Y-ns.E                              ⍝ Fitted values
         ⍝ ns.H←{⍵+.×C+.×⍉⍵}(~⍺)↓[1]1,X                ⍝ Hat matrix
     ns.H←{⍵+.×C+.×⍉⍵}(~INT)↓[1]1,X                ⍝ Hat matrix
     ns.R←ns.E÷ns.S×sqrt 1-0 0⍉ns.H              ⍝ Studentized Residuals
     XX←##.∆r.x'cor(⍵)'X
     ns.VIF←0 0⍉⌹XX                              ⍝ Variance Inflation Factor
     ns.ConditionNumber←(⌈/÷⌊/)eigenValues XX    ⍝
     ns.f←(ns.B DF ns.S C ⍺)linear               ⍝ Build linear function
        ⍝  ns.VariableNames←NL←'Y','X',¨(≢⍺↓ns.B)↑1↓⎕D
     ns.VariableNames←NL←'Y','X',¨(≢INT↓ns.B)↑1↓⎕D
     ns.Equation←NL buildRegressionEquation ns.B
     ns._Report←⍕⍪'The regression equation is:'⍬ ns.Equation ⍬ ns._Report
     ns.Type←'Regress' ⋄ ns}

 regressW←{
          ⍝ Weighted Least Squares
          ⍝ ⍺ List of weights
          ⍝ ⍵ y X1 ... Xn
     ⎕IO←0 ⋄ ⎕ML←3
     W←⍺ ⋄
     X←1⍪1↓⊃⍉⍵                                        ⍝ Predictor Variables
     N←≢y←↑⍵ ⋄ P←≢⍵                                   ⍝ Response Variable
     XW←X×⍤1⊢W
     B←(⌹XW+.×⍉X)+.×XW+.×y
     BB←X×⍤1 sqrt W
     z←y×sqrt W
     SST←z+.×z
     SSR←B+.×BB+.×z
     SSE←SST-SSR
     k←≢BB ⋄ DF←N-k
     C←⌹XW+.×⍉X
     V←÷W                                               ⍝ Covariance Matrix
     E←y-YHAT←B+.×X                                   ⍝ Residuals
     SSX←1↓1{(⍵+.*2)-⍺×N÷⍨(+/⍵)*2}X             ⍝??
     XBAR←⍵+.×¨⊂⍺÷+/⍺                                    ⍝ Means for each X
     MSE←SSE÷DF                                       ⍝ Mean Square Error
     S←(leastSquares ⍵).S                             ⍝ Get S from non-weighted model.
     SB←sqrt 0 0⍉(S*2)×⌹BB+.×⍉BB                    ⍝ Montgomery p. 237 and 362
     ns←'Regression'buildAnovaTable SSR SSE,k DF,⍪2 0 ⍝ Create ANOVA table
     T←B÷SB                                           ⍝ Test statistic
     P←2×DF tDist probability>|T                      ⍝ p-Value
     YX←y,⍉1↓X
     H←(⍉X)+.×C+.×X                                   ⍝ Hat Matrix
     R←E÷S÷sqrt 1-0 0⍉H                              ⍝ Studentized Residuals
     VIF←0 0⍉⌹XX←corr 1↓↓X                          ⍝ Variance inflation factor
     ConditionNumber←(⌈/÷⌊/)eigenValues XX            ⍝ Condition Number
     PR←'B,T,P,N,XBAR,SSX,DF,S,MSE,SB,E,YHAT,YX,C,VIF,R,H' ⍝ Put additional
     PR,←',ConditionNumber'
     _←'ns'⎕NS toNestedVector PR                      ⍝  properties into namespace
     ns
 }

 residuals←{
      ⍝ ∇Residuals
      ⍝ ∊ Regression Analysis, Montgomery & Scott pp. 109-118
      ⍝ ⍺   Parameter List
      ⍝ ⍵ = Dependent and Independent Variables y, x[1], x[2] ... x[n]
      ⍝ ← = Coefficient Estimates (float) b[0], b[1], ... b[n]
      ⍝ ← = Standard Errors (float)
      ⍝ ← = t-Values (float)
      ⍝ ← = p-Values (float)
     ⎕ML←3 ⋄ ⎕IO←0
     ⍺←⍬
     b←⍺{⍺≡⍬:regress ⍵ ⋄ ⍺}⍵
     X←1⍪1↓⊃⍉⍵                             ⍝ Predictor Variables
     n←≢y←↑⍵
     y-b+.×X
 }

 runRegress←{
     ⎕IO←0 ⋄ ⎕ML←3
     int←↑⍺ ⋄ W←⍵
     X←(~int)↓1⍪1↓⊃⍉W                                 ⍝ Predictor Variables
     N←≢y←↑W                                          ⍝ Sample Size
     C←⌹X+.×⍉X                                        ⍝ Covariance Matrix
     k←(¯1×int)+≢B←C+.×Xy←X+.×y                       ⍝ Estimate coefficients
     SSE←(y+.*2)-B+.×Xy                               ⍝ Error Sum of Squares
     E←y-YHAT←B+.×X                                   ⍝ Residuals
     ss←{(⍵+.*2)-⍺×N÷⍨(+/⍵)*2}                        ⍝ SumSquares function
     SST←int ss y                                     ⍝ Total sum of Squares
     SSX←1↓int ss X                                   ⍝ Why is this here?
     XBAR←mean¨W                                      ⍝ Means for each X
     DF←N-k+int                                       ⍝ Degrees of Freedom
     MSE←SSE÷DF                                       ⍝ Mean Square Error
     S←MSE*÷2                                         ⍝ Standard error for regression
     SB←(MSE×0 0⍉C)*÷2                                ⍝ Standard Error for coeffs
     SSR←SST-SSE                                      ⍝ Regression sum of squares
     ns←'Regression'buildAnovaTable SSR SSE,k DF,⍪2 0 ⍝ Create ANOVA table
     ns.AnovaTable←ns.AnovaTable{11::⍺ ⋄ 2≠⍴⍵:⍺       ⍝ If multple reg or no dups, exit
         0∧.=PE←+⌿↑{(2+.*⍨⍵-mean ⍵),¯1+≢⍵}⌸/⌽⍵:⍺      ⍝ Calculate Pure Error SS,DF
         ERR←⍉(⍺[2;1 2]-PE),⍪PE                       ⍝ Split Error into LOF, PE
         P←ERR[;1]fDist prob>F←÷/⍣2⊢ERR               ⍝ Calculate F and P
         Z←'  Lack of Fit' '  Pure Error',(⊢,÷/)ERR   ⍝ Insert Labels
         ⍺[⍳3;]⍪(Z,2 2⍴F P 0 0)⍪⍺[3;]}⍵               ⍝ Insert LOF into ANOVA Table
     T←B÷SB                                           ⍝ Test statistic
     P←2×DF tDist probability>|T                      ⍝ p-Value
     YX←y,⍉1↓X
     Hii←0 0⍉H←(⍉X)+.×C+.×X                           ⍝ Hat Matrix
     R←E÷S×sqrt 1-Hii                                 ⍝ Studentized Residuals
     CooksD←(R*2)÷(≢B)÷(⊢÷1∘-)Hii                     ⍝ Cook's Distance
     PRESS←sumSquares E÷1-Hii                         ⍝ PRESS Statistic
          ⍝VIF←0 0⍉⌹XX←corr int↓↓X                         ⍝ Variance inflation factor
     XX←corr int↓↓X
     VIF←{0::⌊/⍳0 ⋄ 0 0⍉⌹⍵}XX                         ⍝ Variance inflation factor
     ConditionNumber←(⌈/÷⌊/)eigenValues XX            ⍝ Condition Number
     PR←'B,T,P,N,XBAR,SSX,DF,S,MSE,SB,E,YHAT'         ⍝ Put additional
     PR,←',YX,C,VIF,R,H,ConditionNumber,CooksD,PRESS' ⍝
     _←'ns'⎕NS toNestedVector PR                      ⍝  properties into namespace
     ns
 }

 sequence←{
     z←↑⍵
     z,¨↑1↓⍵
 }

 showRegress←{⎕ML←3 ⋄ ⎕IO←0
     XY←⌽⍵.YX                                         ⍝ Get X and Y values
     X y←2↑↓⍉XY
     Z←XY,(XY*2),×/XY                                 ⍝ Get Squares and products
     Z←Z,-⍨\⌽y,⍪⍵.B+.×⍉1,⍪X                            ⍝ Fitted values and Residuals
     Z←Z,(¯1↑[1]Z)*2                                  ⍝ Squared residuals
     T←{⍵×0.00001<|⍵}+⌿Z                              ⍝ Sum all values
     Z←(1+⍳⍵.N),Z                                       ⍝ Include sequence
     H←'n' 'x' 'y' 'x²' 'y²' 'xy' 'yhat' 'e' 'e²'
     Z←H⍪Z⍪(⊂'Total'),T  ⍝ Result
     FT←' ' 'X' 'Y' 'SSX' 'SSY' 'SSXY' ' ' 'SE' 'SSE'
     AV←(2↑T)÷⍵.N
     SS←T[2 3 4]-⍵.N÷⍨{(⍵*2),×/⍵}2↑T
     Z⍪FT,[¯0.5]' ',AV,SS,' ',(T[7]÷⍵.N-2)*0.5 1
 }

 transform←{
     NS←⎕NS''
     NS.Int←1
     NS.Y←⍵.(ChildrenUnder6LivinginPoverty)
     NS.SINGLE←⍵.(Childreninsingleparenthouseholds÷tolerant 100)
     NS.PRE←⍵.(0.01×tolerant PreschoolEnrollmentRatioenrolledages3and4)
     NS}

 transform1←{
     ⍝ User-defined transformation
     ⍝⍵ Namespace
     ⍝← Transformed namespace
     ⍺←1
     X1←{⍵-mean ⍵}⍵.Height
     X2←{×⍨⍵-mean ⍵}⍵.Height
     X3←⍵.ShoeSize
     M←⍵.Sex eq'M'
     XM←(⍵.Sex eq'M')×⍵.ShoeSize
     Y←⍵.Weight
     Int←⍺
     ⎕NS'X1' 'X2' 'X3' 'M' 'XM' 'Y' 'Int'
 }

 weightedLS←{
      ⍝∇  Calculate Linear Weighted Least Squares for indicator response variable
      ⍝⍺  Y values (must be 0's and 1's)
      ⍝⍵  X values
      ⍝←
     NS←regress ⍺ ⍵
     W←{÷⍵×1-⍵}NS.YHAT  ⍝ Weights
     SW←+/W
     SWX←W+.×⍵
     SWX2←W+.×⍵*2
     SWY←W+.×⍺
     SWXY←+/W×⍺×⍵
     NS.B←SWY SWXY⌹2 2⍴SW SWX SWX SWX2
     NS.f←NS.B∘{⍵⊥⌽⍺}
     NS
 }

 xlogistic←{
     ⍝∇ Calculate logistic regression model
     ⍝⍺ Sample Sizes (Optional Vector)
     ⍝⍵ Response Variable (Boolean or Number of Successes)
     ⍝⍵ Predictor Variable
     ⍝← Coefficients
     ⍝← Function
     ⍝⍎ NS←N logistic C X
     ⍝⍎ NS←logistic B X
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ ⍺←1       ⍝ Initialize
     X C N←⍺{⍺≢1:(⌽⍵),⊂⍺       ⍝ C is Successes
         Y X←⍵                 ⍝ N is Sample Sizes
         ↓⍉X{⍺,(+/⍵),≢⍵}⌸Y}⍵   ⍝ X is Predictor Variable
     P←C÷N                     ⍝ Proportion Failing
     PS←ln P÷1-P               ⍝ Transformed logistic response
     W←N×P×1-P                 ⍝ Weights (inverse variance)
     M←2 2⍴1 2 1/W+.×1,X,⍪X*2  ⍝ Create 2x2 matrix
     V←(W+.×PS),+/W×X×PS       ⍝ Vector
     NS←⎕NS''
     NS.B←V⌹M                  ⍝ Int, Slope Parameters
     NS.f←NS.B∘{÷1+*-⍵⊥⌽⍺}     ⍝ Function
     NS
 }

 buildNL←{⎕IO←0 ⋄ ⎕ML←3
     g←⍺⍺ ⋄ CR←↓⎕CR'g' ⍝ Get function
     CR←deleteExcessBlanks¨CR
     CR←CR/⍨{∨/'NS.'⍷⍵}¨CR
     CR←(~↑¨'NS.Int←'∘⍷¨CR)/CR              ⍝ Remove Int
     NM XP←↓⍉{⍵↓⍨1+⍵⍳'.'}¨⊃(CR≠'←')⊂¨CR     ⍝Parse NMS Expr
    ⍝ J←{1⍳⍨'mean'⍷⍵}¨XP
⍝             J+5↓¨XP
 ⍝⍝⍝⍝⍝⍝⍝⍝⍝ Build model function ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝

     D←∊{' ',(~2∨/0,≠\⍵='''')/⍵}¨1↓XP ⍝ Eliminate Quotes
     B←D∊⎕A,⎕UCS 32+⎕UCS ⎕A           ⍝ Find Names
     VN←{∪(⎕A∊⍨↑¨⍵)/⍵}B⊂D             ⍝ Unique Variable Names
     CR←⊂'h←{⍵≡''Type'':''Linear'''
    ⍝CR,←⊂'⍺←0'
    ⍝CR,←⊂'2=⍴⍴⍵:⍺∇⍤1⊢⍵'              ⍝ Multiple inputs?
     CR,←'⍺←0'(⍕VN'←⍵')
     ⍝CR,⊂(⍕VN'←⍵')                    ⍝ Assign names
     CR,←1↓NM{⍺,'←',⍵}¨XP             ⍝ Do the transformation
     X←{⍵[⍋⍵]}1↓NM                    ⍝ Sort the names
     f←(⍵.(B DF S C),1)linear
     CR,←⊂'⍺ ⍺⍺ ',⍕X,'}'
     R←⎕FX CR
     ⍵.g←f h
     ⍵
 }


 linear←{
⍝⍟ Copyright (C) 2018 by Stephen M. Mansour
⍝∇ Linear function
⍝∊ Regression
⍝⍺:  Confidence level
⍝⍺:  Variant 0=function 1=conf  2=pred
⍝⍺⍺: Coefficients
⍝⍺⍺: Degrees of Freedom
⍝⍺⍺: Standard Error
⍝⍺⍺: Covariance Matrix
⍝⍺⍺: Include intercept?
⍝⍵:  Array
⍝←:  b0+(b1×x1)+...+(bn×xn)
⍝    linear
     ⎕IO←0 ⋄ ⎕ML←3
     B DF S C int←⍺⍺                    ⍝ Unpack operator
     ⍵≡'Type':'Linear'                  ⍝ Function type is linear
     ⍺←0                                ⍝ Default
     0J1=↑⍺:'Nonce Error'               ⍝ Show interval? Not yet implemented.
     CL A←2↑⍺                           ⍝ Action, Conf Level
     B1←(int~1),B                       ⍝ Coefficients
     (2=≢B1)∧1<≢,⍵:⊃⍺∘∇¨⍵               ⍝ Simple regression multiple inputs
     1<⍴⍴⍵:⊃⍺∘∇¨↓⍵                      ⍝ Multiple regression multiple input
     Y←B1+.×⍉int,⍤(2<≢B1)⊢⍵             ⍝ Function
     A=0:Y
     d←(S*2)×(A-1)+x+.×C+.×x←(int~0),⍵  ⍝ Calculate standard error(s)
     t←DF tDist criticalValue<0.5×1-CL  ⍝ Critical value of t-distribution
     Y(-,+)t×d*÷2}


 pairwise←{
 ⍝⍺  Optional symbol to
 ⍝⍺⍺ Any dyadic (preferably commutative) function or a character string
 ⍝⍵  Any vector whose elements are in the domain of ⍺⍺
 ⍝←  A vector of shape 2!≢⍵
 ⍝⍎     ,pairwise 'ABCD'
 ⍝⍎ AB  AC  BC  AD  BD  CD
 ⍝⍎     ×pairwise 1 2 3 4 5
 ⍝⍎ 2 3 6 4 8 12 5 10 15 20
 ⍝⍎      ',' pairwise 'ABCD'
 ⍝⍎ A,B  A,C  B,C   A,D   B,D   C,D
 ⍝⍎      ''  pairwise 'Bob' 'Joe 'Steve'
 ⍝    ┌─────────┬───────────┬───────────┐
 ⍝    │┌───┬───┐│┌───┬─────┐│┌───┬─────┐│
 ⍝    ││Bob│Joe│││Bob│Steve│││Joe│Steve││
 ⍝    │└───┴───┘│└───┴─────┘│└───┴─────┘│
 ⍝    └─────────┴───────────┴───────────┘
     ⎕IO←0 ⋄ ⎕ML←3
     1≥≢⍵:0⍴⊂⍺⍺⍨↑⍵
     d←⍵ ⋄ op←⍺⍺ ⋄ ⍺←⍬ ⋄ a←⍺
    ⍝ 2=⎕NC'op':{0=≢op:⍺ ⍵ ⋄ ⍺ op ⍵}∇∇ ⍵
     2=⎕NC'op':{0=≢op:⍺ ⍵ ⋄ ⍺,op,⍵}∇∇ ⍵
     0<≢a:{⍺,a,⍵}∇∇ ⍵
     op{⍺←0↑d
         z←⍺,(⍵↑d)⍺⍺¨⍵⌷d
         ⍵≥¯1+≢d:z
         z ∇ ⍵+1}1
 }


 regress←{
 ⍝∇  Regress Operator
 ⍝⍟  Copyright 2023 by Stephen Mansour
 ⍝∊  Operator
 ⍝⍺  Y         Yv Respose Variable
 ⍝⍺⍺ YName     Cv Name of response variable
 ⍝⍺⍺ f         f  Transformation Function
 ⍝⍵  X         Xv Predictor Variable
 ⍝⍵  Xn        Xn List of Predictor Variables
 ⍝⍵  DataBase W  Contains Predictor and Response Variables
 ⍝← Model W Contains Coefficients, AnovaTable, R-Squared, predictor function f
 ⍝⍕ Model←Y[f]|transform|YName regress X|Xn|Database
 ⍝⍎ report Height regress ShoeSize              ⍝ Simple Regression
 ⍝⍎ report Weight regress Height ShoeSize       ⍝ Multiple Regression
 ⍝⍎ report Height regress ShoeSize (Sex eq 'M') ⍝ Indicator Variable
 ⍝⍎ report Height quad regress ShoeSize         ⍝ Quadratic regression
 ⍝⍎ report Weight cubic regress ShoeSize        ⍝ Cubic regression
 ⍝⍎ report Y ln regress X                       ⍝ Log transform
 ⍝⍎ report transform regress DataBase           ⍝ User-defined transformation
     ⍺←'' ⋄ ⎕IO←0 ⋄ ⎕ML←1
     _←⎕EX'#.∆NS'
     FN←⊃3=⎕NC'⍺⍺'                       ⍝ Is left operand a function?
     DB←⊃9=⎕NC'⍵'                        ⍝ Is right argument a database?
     Int X←{(1=≡⍵)∨~(⊂⊃⍵)∊0 1:1 ⍵ ⋄ 2=≢⍵:⍵
         (⊃⍵)(1↓⍵)}⍵                     ⍝ Include intercept?
     2=⍴⍴X:Int leastSquares ⍺⍺,X         ⍝ Matrix right argument?
     ~FN:(,⍺⍺)(Int regressInd 1)X
     FN∧~DB:⍺ ⍺⍺{U←3 ⍺⍺ 4                ⍝ Test operand
         U=4:⍺(Int regressPoly)⍵         ⍝ Polynomial (⊥)
         U=5:⍺(⍺⍺ 1)(Int regressPoly)⍵   ⍝ Cubic/Quadratic
         U∊0 1:⍺(Int regressBin)⍵        ⍝ Binary regression (= or ≠)
         ⍺(⍺⍺ vst Int)⍵                  ⍝ Variance stabilizing transformation
     }X
     DB∧~FN:(,⍺⍺)(Int regressInd 1)⍵        ⍝ Don't Ignore Character Fields
     FN∧⍺≢'':⍺ ⍺⍺{
         X←⍵.(⍎⍕(' '~⍨¨↓⎕NL 2)~⊂⍺)          ⍝ Unpack predictor variables
         (⍵⍎⍺)(⍺⍺ regress)X            ⍝ Apply regress
       ⍝  U←2 ⍺⍺ 4               ⍝ Treat char fields as
⍝         U=4:⍺(Int regressPoly)⍵
⍝         U=5:⍺(⍺⍺ 1)(Int regressPoly)⍵  ⍝ Cubic/Quadratic
⍝         U∊0 1:⍺(Int regressBin)⍵
     }⍵
     FN∧⍺≡'':⍺⍺{NS←⍺⍺ ⍵                  ⍝ If user operand, apply transformation
         V←⊆variables NS                 ⍝ Get variable names
         V←⍕'Y',V~,¨'Y' 'Int'            ⍝ Put response variable first
         NS←NS.Int leastSquares NS V     ⍝ Do regression
         ⍵ ⍺⍺ buildNL NS}X               ⍝ Build non-linear function

     Y←,⍺⍺{2=⎕NC'⍺⍺':⍺⍺ ⋄ ⍵}⍺            ⍝ Get name of response variable
     V←⊆variables ⍵                      ⍝ Get transformation variable names
     V←⍕Y,V~'Int'Y                       ⍝ Put response variable first
     ⍵.∆←⍺⍺¨⍣FN⊢⍵⍎V                      ⍝ Apply transformation function to all variables
     _←⍵⍎V,'←∆'                          ⍝ Reassign transformed data function names
     _←⍵.(⎕EX'∆')                        ⍝ Clean up garbage
     I←{0::1 ⋄ ⊃⍵.Int}⍵                  ⍝ Include intercept?
     I leastSquares ⍵ V                  ⍝ Do regression
 }


 regress2←{
 ⍝⍺  [Optional] Name of response variable
 ⍝⍺⍺ transform
 ⍝⍵  Namespace
     ⍺←'' ⋄ ⎕IO←0 ⋄ ⎕ML←1
     FN←⊃3=⎕NC'⍺⍺'              ⍝ Is left operand a function?
     DB←⊃9=⎕NC'⍵'               ⍝ Is right argument a database?
     Int X←{¯2=≡⍵:(⊃⍵)(1↓⍵) ⋄ 1 ⍵}⍵        ⍝ Include intercept?
     FN⍱DB:Int leastSquares ⍺⍺,⍉↑,⊆X ⍝ Old method:  Y regress X
     FN∧~DB:⍺ ⍺⍺{U←2 ⍺⍺ 4       ⍝ Test operand
         U=4:             ⍝ Polynomial (⊥)
         U∊0 1:'Indicator'      ⍝ Indicator (= or ≠)
         (U=2)∧0=⍺⍺ 1:'Log'     ⍝ Log transform
         U=16:'b×x*a'           ⍝ Y = b×x*a
         U=0.5:'÷'              ⍝ Inverse transform
         U=2:'sqrt'             ⍝ Square root transform
         'arcsin'               ⍝ Arcsin transform
     }X
     FN∧⍺≡'':⍺⍺{NS←⍺⍺ ⍵        ⍝ If user operand, apply transformation
         V←⊆variables NS       ⍝ Get variable names
         V←⍕'Y',V~,¨'Y' 'Int'  ⍝ Put response variable first
         NS.Int leastSquares NS V}X ⍝ Do regression
     Y←,⍺⍺{2=⎕NC'⍺⍺':⍺⍺ ⋄ ⍵}⍺  ⍝ Get name of response variable
     V←⊆variables ⍵            ⍝ Get transformation variable names
     V←⍕Y,V~'Int'Y             ⍝ Put response variable first
     ⍵.∆←⍺⍺¨⍣FN⊢⍵⍎V            ⍝ Apply transformation function to all variables
     _←⍵⍎V,'←∆'                ⍝ Reassign transformed data function names
     _←⍵.(⎕EX'∆')              ⍝ Clean up garbage
     I←{0::1 ⋄ ⍵.Int}⍵         ⍝ Include intercept?
     I leastSquares ⍵ V        ⍝ Do regression

 }


 regressBin←{
⍝∇ Binary Regression

⍝⍺  Y or Name  of Binary Responses
⍝⍺⍺ Int
⍝⍵  C X1 .. Xn   or Database
     ⎕ML←1 ⋄ ⎕IO←0                  ⍝ Default values
     DB←9=⎕NC'⍵'                    ⍝ Is right argument a database?
     2∧.=(≡,≢,≢∘⊃∘⌽)⍵:⍺{        ⍝ If initial guess included
         Y←⍺
        ⍝ B←⊃⌽⍵ ⋄ X←Int,⍪⍉⊃¯1↓⍵     ⍝ Separate X from guess
         X B←⍵ ⋄ ⎕PP←5
        ⍝ B JBI H Pi←(Y,⍪X)logistic⍣≡B ⍝ MLE estimatation of parameters
         B JBI H Pi←(Y,⍪X)logistic⍣100⊢B ⍝ MLE estimatation of parameters
         DF←⍺-⍥≢B
         S←sqrt DF÷⍨sumSquares Y-Pi
         NS←⎕NS''
         NS.f←B DF S JBI 1 linear
         NS.g←{⍵≡'Type':'Linear'
             ⍺←0 ⋄ CL A←2↑⍺
             A=0:÷1+*-f ⍵
             ÷1+*-⍺ f ⍵}
         SE←sqrt 0 0⍉JBI
         Z←B÷SE
         P←2×normal prob>|Z
         Coeff←'*' 'Int' 'X','Coeff' 'SE' 'Z' 'P'⍪B,SE,Z,⍪P
         ⍝Freq←frequency show ⍺
         Freq←⍕(⊂'Total')@(⊂3 0)⊢'Value' 'Count'⍪(⊖⍪+⌿)frequency ⍺
         OddsRatio←*B[1]
         LogLikelihood←(⍺+.×⍟Pi÷1-Pi)++/⍟1-Pi ⍝ Log Likelihood
         Expr←⍕'Y←÷1+exp -',B[0],'+(',B[1],'× X) + E'
         V←2 2⍴'Odds Ratio:'OddsRatio'Log Likelihood:'LogLikelihood
         NS._Report←⍕⍪'The regression equation is:' ' 'Expr' 'Freq' 'Coeff' 'V
         NS.Type←'Regress'
         _←'NS'⎕NS'B' 'Freq' 'OddsRatio' 'LogLikelihood' 'Coeff' 'Expr'
         NS
     }⍵
     D←DB{⍺:⍵⍎¨variables ⍵  ⍝ Obtain data values
         ↑¨⊆⍵}⍵             ⍝
     B←{0≢⊃0⍴⍵}¨D           ⍝ Find character fields
     NS←(∨/B){⍺:⊃makeIndicator¨⍵ ⋄ ⎕NS''}B/D

     Z←⍺⍺ leastSquares ⍺ ⍵
     W←{÷⍵×1-⍵}Z.YHAT
     Y←⍺(DB{⍺⍺:⍵⍎⍺ ⋄ ⍺})⍵
     W leastSquares Y,⍥⊆⍵

⍝     ~DB:W leastSquares Y,⍥⊆X
     Z.g←{⍵≡'Type':'Linear'

     }

  ⍝
⍝     V←(variables ⍵)~⊂⍺
⍝     W leastSquares ⍵(⍕⍺ V)



     ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝ Create function???
     Z.g←Deg M{⍵≡'Type':'Linear'
         ⍺←0                                ⍝ Default
         CL A←2↑⍺                           ⍝ Action, Conf Level
         D M←⍺⍺
         X←D{1=≢M:⍵∘.*(~⎕IO)+⍳⍺
             ∊(×/⍵)pairwise↓⍵∘.*1 2}⍵-M
         A=0:f X
         ⍺ f X}




 }


 regressInd←{
⍝∇ Regress using indicator  variables
⍝
⍝⍺  Y or Name
⍝⍺⍺ Int
⍝⍵  C X1 .. Xn   or Database
⍝⍵⍵ 1=Character vectors are indicator variables
⍝⍵⍵ 0=Character vectors are ignored
⍝←
     ⎕ML←1 ⋄ ⎕IO←0          ⍝ Default values
     DB←9=⎕NC'⍵'            ⍝ Is right argument a database?
     W←DB{⍺:⍵⍎¨variables ⍵  ⍝ Obtain data values
         ↑¨⊆⍵}⍵             ⍝
     B←{0≢⊃0⍴⍵}¨W           ⍝ Find character fields
    ⍝B[0]≥←∨/B,DB           ⍝ Use 1st numeric field as Ind?
     Y←⍺{0=⊃0⍴⍺:⍺ ⋄ ⍵⍎⍺}⍵   ⍝ Response Variable
   ⍝  U←Y∘{{⊃⍵[⍋⍵[;1];0]}⍺{(⊂⍺),⍥,(+/÷≢)⍵}⌸⍨⍵}¨B/W  ⍝ Make Minimum mean base case
     U←Y∘{0::''
         M←⍺{⍺,⍥⊂(+/÷≢)⍵}⌸⍨⍵

         ↑⊃M[⍋M[;1];0]}¨B/W

    ⍝ NS←(⍵⍵∧∨/B){⍺:⊃makeIndicator¨⍵ ⋄ ⎕NS''}B/W
     NS←(⍵⍵∧∨/B){⍺:⊃U makeIndicator¨⍵ ⋄ ⎕NS''}B/W ⍝ Put indicator variables into namespace
     X←{⊃⍣(1=≢⍵)⊢⍵}(~B)/W
     NMS←DB B{DB B←⍺
         DB:(~B)/variables ⍵
         'X',¨(B+.=0)↑1↓⎕D}⍵
     _←⍎⍕NMS'←X'            ⍝ Else assign to names in char field
     _←'NS'⎕NS NMS
     NMS←variables NS
     DB:⍺⍺ leastSquares NS,⊂⍕NMS(⊢,⊣~⊢)⊂⍺
     NS.Y←⍺
     Z←⍺⍺ leastSquares NS(⍕'Y',NMS)

     ⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝⍝ Create function???
     ⍝Z.g←Deg M{⍵≡'Type':'Linear'
⍝         ⍺←0                                ⍝ Default
⍝         CL A←2↑⍺                           ⍝ Action, Conf Level
⍝         D M←⍺⍺
⍝         X←D{1=≢M:⍵∘.*(~⎕IO)+⍳⍺
⍝             ∊(×/⍵)pairwise↓⍵∘.*1 2}⍵-M
⍝         A=0:f X
⍝         ⍺ f X}
     Z



 }


 regressPoly←{
⍝Polynomial regression
⍝⍺  Y D
⍝⍺⍺ Int
⍝⍵  X1 [X2]
⍝←
     ⎕ML←1 ⋄ ⎕IO←0 ⋄ ⎕PP←5                              ⍝
     Y Deg←2↑(⊆⍺),2                                     ⍝ Separate Y and degree
     (Deg=3)∧2=|≡⍵:Y{X Knots←⍵                          ⍝ Cubic Spline?
         Y X←⍺ removeNulls X
         Z←leastSquares Y,(X∘.*1 2 3),(0⌈X∘.-Knots)*3
         V←'X' 'X*2' 'X*3',⊆{'(0⌈X-',(⍕⍵),')*3'}¨Knots
         Z.Equation←⊃⌽V{Z.Equation←(⍵ ⎕R ⍺)Z.Equation}¨1↓Z.VariableNames
         Z._Report←(Z._Report⌈⍥⍴⍉⍪Z.Equation)↑Z._Report ⍝ Expand report to fit equation
         Z._Report[2;⍳⍴Z.Equation]←Z.Equation           ⍝ Insert polynomial equation
         Z.g←Knots{⍵≡'Type':'Linear'                    ⍝ Build polynomial function
             ⍺←0 ⋄ CL A←2↑⍺                             ⍝ Action, Conf Level
             Knots←⍺⍺
             X←(⍵∘.*1 2 3),(0⌈⍵∘.-Knots)*3
             A=0:f X
             ⍺ f X}
         Z}⍵
     Y W←Y removeNulls ⍵                                ⍝
     NS←Deg makePolynomial{2=≢⊆⍵:⊃⍺⍺/⍵                  ⍝
         ⍺ ⍺⍺ ⍵}W                                       ⍝
     NS.Y←Y                                             ⍝
     NMS←variables NS                                   ⍝
     Z←⍺⍺ leastSquares NS,⊂⍕NMS(⊢,⊣~⊢)⊂,'Y'             ⍝
     M←mean¨⊆⍵                                          ⍝ Center the data
     Z.g←Deg M{⍵≡'Type':'Linear'                        ⍝ Build polynomial function
         ⍺←0                                            ⍝ Default
         CL A←2↑⍺                                       ⍝ Action, Conf Level
         D M←⍺⍺
         X←D{1=≢M:⍵∘.*(~⎕IO)+⍳⍺
             ∊(×/⍵)pairwise↓⍵∘.*1 2}⍵-M
         A=0:f X
         ⍺ f X}
     rAv:Z
     V←⊆M{'X',(⍕⍵),'-',⍕⍺}¨1+⍳⍴M                        ⍝ Variable names
     E←V∘.{⍵=1:⍺ ⋄ '(',⍺,')*',⍕⍵}1+⍳Deg                 ⍝ Expression Names
     XY←'×'pairwise'(',¨V,¨,')'                         ⍝
     E←XY{0=≢⍺:,⍵ ⋄ ⍕¨⍵[0;],⍺,⍵[1;]}E                   ⍝
     Z.Equation←⊃⌽E{Z.Equation←(⍵ ⎕R ⍺)Z.Equation}¨1↓Z.VariableNames
     Z._Report←(Z._Report⌈⍥⍴⍉⍪Z.Equation)↑Z._Report     ⍝ Expand report to fit equation
     Z._Report[2;⍳⍴Z.Equation]←Z.Equation               ⍝ Insert polynomial equation
     Z
 }


 taguchi←{
     d←⍵
     ⍺⍺{⍺←⊂0↑d
         z←⍺,⍺ ⍺⍺¨⍵⊃d
         ⍵≥¯1+≢d:z
         z ∇ ⍵+1}0
 }


 vst←{
⍝ ∇ Variance stablizing transformation
⍝ Copyright 2023 by Stephen Mansour
⍝ ⍺ Response variable
⍝ ⍺⍺ transformation function
⍝ ⍵⍵ Intercept
⍝ ⍵ Predictor variables
⍝     0=⍟;1=⊢;2=sqrt;3=arcsin sqrt;4=×*;5=÷sqrt;6=÷
     ⎕IO←0 ⋄ ⎕ML←1 ⋄ ⎕PP←5
     I←1+0.25 0.5 0.52 1 2 4⍸¯1⌈⍺⍺ 0.25
     F←I⊃'ln' '' 'sqrt' 'arcsin sqrt' 'exp' '÷sqrt' '÷'
     insert←{D←↓⍵ ⋄ (2⊃D)←⍺ ⋄ ↑D}
     I=4:⍵⍵{NS←⍺ leastSquares⍟⍵
         NS.g←{⍵≡'Type':'Linear'
             ⍺←0 ⋄ CL A←2↑⍺
             A=0:*f⍟⍵
             *⍺ f⍟⍵}
         NS.Equation←(⊃NS.VariableNames),'←',(⍺/⍕*⊃NS.B)
         NS.Equation,←(~⍺)↓∊(1↓NS.VariableNames){'×(',⍺,'*',(⍕⍵),')'}¨⍺↓NS.B
         NS.Equation,←'+E'
        ⍝ NS.(B[0]←*B[0])
         NS←{1=≢⍵.B:⍵ ⋄ ⍵.B[0]←*⍵.B[0] ⋄ ⍵}NS
         NS.(_Report←Equation insert _Report)
         NS}⍺ ⍵
     NS←⍵⍵ leastSquares(⍺⍺ ⍺)⍵
     INV←I⊃'exp' '⊣' '×⍨' '×⍨sin' 'ln' '×⍨÷' '÷'
     NS.g←(⍎INV){⍵≡'Type':'Linear'              ⍝ Build function
         ⍺←0                                ⍝ Default
         CL A←2↑⍺                           ⍝ Action, Conf Level
         A=0:⍺⍺ f ⍵
         ⍺⍺⊢⍺ f ⍵}

     NEW←'←',INV,' '
     NS.Equation←('←'⎕R NEW)NS.Equation
     NS.(_Report←Equation insert _Report)
     NS



 }


:EndNamespace 
